//	GeometryGamesHanzi.c
//
//	Convert traditional Chinese characters to simplified Chinese characters.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "GeometryGamesHanzi.h"

#define	NUM_HANZI	((0x9FA5 - 0x4E00) + 1)

//	This forward declaration lets the function SimplifiedHanzi()
//	refer to the array, while postponing its lengthly definition
//	until later.
static Char16 gSimplifiedHanzi[NUM_HANZI + 1];

Char16 SimplifiedHanzi(Char16 aCharacter)
{
	signed int	theOffset;

	//	Convert aCharacter to an offset within the Unicode 4.0
	//	CJK Unified Ideographs range 0x4E00 through 0x9FA5 ('一' through '龥').
	theOffset = aCharacter - 0x4E00;
	
	//	If aCharacter does indeed lie within the CJK range,
	//	return its simplified equivalent.
	//	In many cases this is the same character,
	//	either because the simplified and traditional forms are the same,
	//	or because aCharacter was a simplified form to begin with.
	//
	if (theOffset >= 0 && theOffset < NUM_HANZI)
		return gSimplifiedHanzi[theOffset];
	else
		return aCharacter;
}

static Char16 gSimplifiedHanzi[NUM_HANZI + 1] =	//	allow space for (unneeded) terminating zero in initializer
u"一丁丂七丄丅丆万丈三上下丌不与丏丐丑丒专且丕世丗丘丙业丛东丝丞丢丠両丢丣两严并丧丨丩个丫丬中丮丯丰丱串丳临丵丶丷丸丹为主丼丽举丿"
u"乀乁乂乃乄久乆乇么义乊之乌乍乎乏乐乑乒乓乔乕乖乗乘乙乚乛乜九乞也习乡乢乣乤乥书乧乨乩乪乫乬乭乮乯买乱乲乳乴乵乶乷乸乹乺乻乼乽干乿"
u"亀亁乱亃亄亅了亇予争亊事二亍于亏亐云互亓五井亖亗亘亘亚些亜亝亚亟亠亡亢亣交亥亦产亨亩亪享京亭亮亯亰亱亲亳亴亵亶亷亸亹人亻亼亽亾亿"
u"什仁仂仃仄仅仆仇仈仉今介仌仍从仏仐仑仒仓仔仕他仗付仙仚仛仜仝仞仟仠仡仢代令以仦仧仨仩仪仫们仭仮仯仰仱仲仳仴仵件价仸仹仺任仼份仾仿"
u"伀企伂伃伄伅伆伇伈伉伊伋伌伍伎伏伐休伒伓伔伕伖众优伙会伛伜伝伞伟传伡伢伣伤伥伦伧伨伩伪伫伬伭伮伯估伱伲伳伴伵伶伷伸伹伺伻似伽伾伿"
u"佀佁佂佃佄佅但伫布佉佊佋佌位低住佐佑佒体占何佖佗佘余佚佛作佝佞佟你佡佢佣佤佥佦佧佨佩佪佫佬佭佮佯佰佱佲佳佴并佶佷佸佹佺佻佼佽佾使"
u"侀侁侂侃侄侅来侇侈侉侊例侌侍侎侏侐侑侒侓侔侕仑侗侘侙侚供侜依侞侟侠価侢侣侤侥侦侧侨侩侪侫侬侭侮侯侰侱侲侳侴侵侣局侸侹侺侻侼侽侾便"
u"俀俣系促俄俅俆俇俈俉俊俋俌俍俎俏俐俑俒俓伣俕俖俗俘俙俚俛俜保俞俟侠信俢俣俤俥俦俧俨俩俪俫私俭修俯俰具俲俳俴俵俶俷俸俹俺俻俼俽俾俿"
u"伥倁倂倃倄倅俩倇俫仓倊个倌倍倎倏倐们倒倓倔倕幸倗倘候倚倛倜倝倞借倠倡倢仿値倥倦倧倨倩倪伦倬倭倮倯倰倱倲倳倴倵倶倷倸倹债倻值倽倾倿"
u"偀偁偂偃偄偅偆假偈伟偊偋偌偍偎偏偐偑偒偓偔偕偖偗偘偙做偛停偝偞偟偠偡偢偣偤健偦偧偨偩偪偫偬偭偮偯偰偱偲偳侧侦偶偷偸偹偺偻偼伪偾偿"
u"傀傁傂傃傄傅傆傇傈傉傊傋傌傍傎傏傐杰傒傓傔傕伧傗伞备傚傛傜傝傞傟傠傡家傣傤傥傦傧储傩傪傫催佣傮偬傰傱傲传伛债傶伤傸傹傺傻傼傽倾傿"
u"僀僁偻僃僄仅僆戮僈佥僊僋僌働僎像僐侨僒僓僔仆僖僗僘僙僚僛僜僝伪僟僠僡僢僣僤侥僦僧偾僩僪僫僬僭僮僯僰雇僲僳僴僵僶僷僸价僺僻僼僽僾僿"
u"仪儁侬儃亿儅儆儇侩俭儊儋儌儍儎儏傧儑儒儓俦侪儖儗尽儙儚儛儜儝儞偿儠儡儢儣儤儥儦儧儨儩优儫儬儭儮儯儰儱储儳儴儵儶俪㑩儹傩傥俨儽儾儿"
u"兀允兂元兄充兆凶先光兊克兑免兎兏児兑儿兓兔兕兖兖兘兙党兛兜兝兞兟兠兡兢兣兤入兦内全两兪八公六兮兯兰共兲关兴兵其具典兹兺养兼兽兾兿"
u"冀冁冂冃冄内円冇冈冉册冋册再冎冏冐冑冒冓冔冕冖冗冘写冚军农冝冞冟冠冡冢冣冤冥冦冧冨冩幂冫冬冭冮冯冰冱冲决冴况冶冷冸冹冺冻冼冽冾冿"
u"净凁凂凃凄凅准凇净凉凊凋凌冻凎减凐凑凒凓凔凕凖凗凘凙凚凛凛凝凞凟几凡凢凣凤凥処凧凨凩凪凫凬凭凮凯凰凯凲凳凴凵凶凷凸凹出击凼函凾凿"
u"刀刁刂刃刄刅分切刈刉刊刋刌刍刎刏刐刑划刓刔刕刖列刘则刚创刜初刞刟删刡刢刣判别刦刧刨利删别刬刭刮刯到刱刲刳刴刵制刷券刹刺刻刼刽刾刿"
u"剀剁剂剃刭剅剆则剈剉削克剌前刹剏剐剑剒剓剔剕剖刬剘剙剚刚剜剥剞剟剠剡剢剣剤剥剦剧剨剩剪剫剬剭剐副剰剱割剳剀创剶铲剸剹剺剻剼剽剾剿"
u"劀劁劂划劄劅劆剧劈刘刽劋刿剑劎㓥劐剂劒劓劔劕劖劗劘劙㔉力劜劝办功加务劢劣劤劥劦劧动助努劫劬劭劮劯劰励劲劳労劵劶劷劸効劺劻劼劽劾势"
u"勀劲勂勃勄勅勆勇勈勉勊勋勌勍勎勏勐勑勒勓勔动勖勖勘务勚勋勜胜劳募勠勡势勣勤勥勦勧勨勚勪勫勬勭勮勯勰劢勲勋勴励勶勷劝勹勺匀勼勽勾勿"
u"匀匁匂匃匄包匆匇匈匉匊匋匌匍匎匏匐匑匒匓匔匕化北匘匙匚匛匜匝匞匟匠匡匢匣匤匥匦匧匨匩匪匫匬匦匮汇匰匮匲匳匴匵匶匷匸匹区医匼匽匾匿"
u"区十卂千卄卅卆升午卉半卋卌卍华协卐卑卒卓协单卖南単卙博卛卜卝卞卟占卡卢卣卤卥卦卧卨卩卪卫卬卭卮卯印危卲即却卵卶卷卸恤卺却卼卽卾卿"
u"厀厁厂厃厄厅历厇厈厉厊压厌厍厎厏厐厑厒厓厔厕厖厗厘厍厚厛厜厝厞原厕厡厢厣厤厥厦厧厨厩厪厫厬厌厮厯厰厱厉厳厣厵厶厷厸厹厺去厼厽厾县"
u"叀叁参参叁叅叆叇又叉及友双反収叏叐发叒叓叔叕取受变叙叚叛叜叝叞叟叠叡丛口古句另叧叨叩只叫召叭叮可台叱史右叴叵叶号司叹叺叻叼叽叾叿"
u"吀吁吂吃各吅吆吇合吉吊吋同名后吏吐向咤吓吔吕吖吗吘吙吚君吜吝吞吟吠吡吣吣吤吥否吧吨吩吪含听吭吮启吰吱吲吴吴吵呐吷吸吹吺吻吼吽吾吿"
u"呀呁吕呃呄呅呆呇呈呉告呋呌呍呎呏呐呑呒呓呔呕呖呗员呙呚呛呜呝呞呟呠呡呢呣呤呥呦呧周呩呪呫呬呭呮呯呰呱呲味呴呵呶呷呸呹呺呻呼命呾呿"
u"咀咁咂咃咄咅咆咇咈咉咊咋和咍咎咏咐咑咒咓咔咕咖咗咘咙咚咛咜咝咞咟咠咡咢咣咤咥咦咧咨咩咪咫咬咭咮咯咰咱咲咳咴咵咶啕咸咹咺咻呙咽咾咿"
u"哀品哂哃哄哅哆哇哈哉哊哋哌响哎哏哐哑哒哓哔哕哖哗哘哙哚哛哜哝哞哟哠员哢哣哤哥哦哧哨哩哪哫哬哭哮哯哰哱哲哳哴哵哶哷哸哹哺哻哼哽哾哿"
u"唀唁唂唃呗唅唆唇唈唉唊唋唌唍唎唏唐唑唒唓唔唕唖唗唘唙吣唛唜唝唞唟唠唡唢唣唤唥唦唧唨唩唪唫唬唭售唯唰唱唲唳唴唵唶唷念唹唺唻唼唽唾唿"
u"啀啁啂啃啄啅商啇啈啉啊啋啌啍啎问啐啑啒启啔啕啖啗啘啙啚啛啜啝哑启啠啡唡啣啤啥啦啧啨啩啪啫啬啭啮啯啰啱啲啳啴啵啶啷啸啹啺啻啼啽啾啿"
u"喀喁喂喃善喅喆喇喈喉喊喋喌喍㖞喏喐喑喒喓喔喕喖喗喘喙唤喛喜喝喞喟喠喡喢喣喤喥喦喧亮喩丧吃乔喭单喯喰喱哟喳喴喵営喷喸喹喺喻喼喽喾喿"
u"嗀嗁嗂嗃嗄嗅呛啬嗈嗉唝嗋嗌嗍吗嗏嗐嗑嗒嗓嗔嗕嗖嗗嗘嗙呜嗛嗜嗝嗞嗟嗠嗡嗢嗣嗤嗥嗦嗧嗨唢嗪嗫嗬嗭嗮嗯嗰嗱嗲嗳嗴嗵哔嗷嗸嗹嗺嗻嗼嗽嗾嗿"
u"嘀嘁嘂嘃嘄嘅叹嘇嘈嘉嘊嘋嘌喽嘎嘏嘐嘑嘒嘓呕嘕啧尝嘘嘙嘚嘛唛嘝嘞嘟嘠嘡嘢嘣嘤嘥嘦嘧嘨哗嘪嘫嘬嘭唠啸叽嘱嘲嘳嘴哓嘶嘷呒嘹嘺嘻嘼啴嘾嘿"
u"噀噁噂噃噄噅噆噇噈噉噊噋噌噍噎噏噐噑噒嘘噔噕噖噗噘噙㖊噛噜咝噞噟哒噡噢噣噤哝哕噧器噩噪噫噬噭噮嗳噰噱哙噳喷噵噶噷吨当噺噻噼噽噾噿"
u"咛嚁嚂嚃嚄嚅嚆吓嚈嚉嚊嚋哜嚍嚎嚏尝嚑嚒嚓嚔噜嚖嚗嚘啮嚚嚛嚜嚝嚞嚟嚠嚡嚢嚣嚤咽呖嚧咙嚩嚪嚫嚬嚭向嚯嚰嚱亸喾严嚵嘤嚷嚸嚹嚺嚻嚼嚽嚾嚿"
u"啭嗫嚣囃囄冁囆囇呓啰囊囋囌禧囎囏囐嘱囒啮囔囕囖囗囘囙囚四囜囝回囟因囡团団囤囥囦囧囨囩囱囫囬园囮囯困囱囲図围囵囶囷囸囹固囻囼国图囿"
u"圀圁圂圃圄圅圆囵圈圉圊国圌围圎圏圐圑园圆圔圕图圗团圙圚圛圜圝圞土圠圡圢圣圤圥圦圧在圩圪圫圬圭圮圯地圱圲圳圴圵圶圷圸圹场圻圼圽圾圿"
u"址坁坂坃坄坅坆均坈坉坊坋坌坍坎坏坐坑坒坓坔坕坖块坘坙坚坛坜坝坞坟坠坡坢坣坤坥坦坧坨坩坪坫坬坭坮坯坰坱坲坳坴坵坶坷坸坹坺坻坼坽坾坿"
u"垀垁垂垃垄垅垆垇垈垉垊型垌垍垎垏垐垑垒垓垔垕垖垗垘垙垚垛垜垝垞垟垠垡垢垣垤垥垦垧垨垩垪垫垬垭垮垯垰垱垲垳垴埯垶垷垸垹垺垻垼垽垾垿"
u"埀埁埂埃埄埅埆埇埈埉埊埋埌埍城埏埐埑埒埓埔埕埖埗埘埙埚埛埜埝埞域埠垭埢埣埤埥埦埧埨埩埪埫埬埭埮埯采埱埲埳埴埵埶执埸培基埻埼埽埾埿"
u"堀堁堂堃堄坚堆堇堈堉垩堋堌堍堎堏堐堑堒堓堔堕垴堗堘堙堚堛堜埚堞堟堠堡堢堣堤堥堦堧堨堩堪堫堬堭堮尧堰报堲堳场堵堶堷堸堹堺堻堼堽堾堿"
u"塀塁塂塃塄塅塆塇塈塉块茔塌塍塎垲塐塑埘塓塔塕塖涂塘塙冢塛塜塝塞塟塠塡坞塣埙塥塦塧塨塩塪填塬塭塮塯塰塱塲塳塴尘塶塷塸堑塺塻塼塽塾塿"
u"墀墁墂境墄墅墆墇墈墉垫墋墌墍墎墏墐墑墒墓墔墕墖増墘墙墚墛坠墝增墟墠墡墢墣墤墥墦墧墨墩墪墫墬墭堕墯墰墱墲坟墴墵墶墷墸墹墺墙墼墽垦墿"
u"壀壁壂壃壄壅壆坛壈壉壊垱壌壍埙壏壐壑壒压壔壕壖壗垒圹垆壛坛壝坏垄垅壡坜壣壤壥壦壧壨坝壪士壬壭壮壮声壱売壳壴壵壶壷壸壹壶壻壸寿壾壿"
u"夀夁夂夃处夅夆备夈変夊夋夌复夎夏夐夑夒夓夔夕外夗夘夙多夛夜夝夞够够夡梦夣夤伙夦大夨天太夫夬夭央夯夰失夲夳头夵夶夷夸夹夺夻夼夽夹夿"
u"奀奁奂奃奄奅奆奇奈奉奊奋奌奍奎奏奂契奒奓奔奕奖套奘奙奚奛奜奝奞奟奠奡奢奣奤奥奦奥奨奁夺奫奖奭奋奯奰奱奲女奴奵奶奷奸她奺奻姹好奾奿"
u"妀妁如妃妄妅妆妇妈妉妊妋妌妍妎妏妐妑妒妓妔妕妖妗妘妙妚妛妜妆妞妟妠妡妢妣妤妥妦妧妨妩妪妫妬妭妮妯妰妱妲妳妴妵妶妷妸妹妺妻妼妽妾妿"
u"姀姁姂姃姄姅姆姇姈姉姐始姌姗姎姏姐姑姒姓委姕姖姗姘姙姚姛姜姝姞姟姠姡姢姣姤姥奸姧姨姩侄姫姬姭姮姯姰姱姲姳姴姵姶姷姸姹姺姻姼姽姾姿"
u"娀威娂娃娄娅娆娇娈娉娊娋娌娍娎娏娐娑娒娓娔娕娖娗娘娙娚娱娜娝娞娟娠娡娢娣娤娥娦娧娨娩娪娫娬娭娮娯娰娱娲娳娴娵娶娷娸娹娺娻娼娽娾娿"
u"婀娄婂婃婄婅婆婇婈婉婊婋婌婍婎婏婐婑婒婓婔婕婖婗婘婙婚婛婜婝婞婟婠婡婢婣婤婥妇婧婨婩婪婫婬娅婮婯婰婱婲婳婴婵婶婷婸婹婺婻婼婽婾婿"
u"媀媁媂媃媄媅媆媇媈媉媊媋媌媍媎媏媐媑媒媓媔媕媖媗媘媙媚媛媜媝媞媟媠媡媢媣媤媥媦娲媨媩媪媫媬媭媮妫媰媱媲媳媴媵媶媷媸媹媺媻媪妈媾媿"
u"嫀嫁嫂嫃嫄嫅嫆嫇嫈嫉嫊袅嫌嫍嫎嫏嫐嫑嫒嫓嫔嫕嫖妪嫘嫙嫚嫛嫜嫝嫞嫟嫠嫡嫢嫣嫤嫥嫦嫧嫨嫩嫪嫫嫬嫭嫮嫯嫰嫱嫲嫳嫴妩嫶嫷嫸嫹嫺娴嫼嫽嫾婳"
u"妫嬁嬂嬃嬄嬅嬆嬇娆嬉嬊婵娇嬍嬎嬏嬐嬑嬒嬓嬔嬕嬖嬗嬘嫱嬚嬛嬜袅嬞嬟嬠嫒嬢嬣嬷嬥嬦嬧嬨嬩嫔嬫嬬嬭嬮嬯婴嬱嬲嬳嬴嬵嬶嬷婶嬹嬺嬻嬼嬽嬾嬿"
u"孀孁孂娘孄孅孆孇孈孉孊孋娈孍孎孏子孑孒孓孔孕孖字存孙孚孛孜孝孞孟孠孡孢季孤孥学孧孨孩孪孙孬孭孮孯孰孱孲孳孴孵孶孷学孹孺孻孼孽孾孪"
u"宀宁宂它宄宅宆宇守安宊宋完宍宎宏宐宑宒宓宔宕宖宗官宙定宛宜宝实実宠审客宣室宥宦宧宨宩宪宫宬宭宫宯宰宱宲害宴宵家宷宸容宺宻宼宽宾宿"
u"寀寁寂寃寄寅密寇寈寉寊寋富寍寎寏寐寑寒寓寔寕寖寗寘寙寚寛寜寝寞察寠寡寝寣寤寥实宁寨审寪写宽寭寮寯寰寱寲寳寴宠宝寷寸对寺寻导寽対寿"
u"尀封専尃射克将将专尉尊寻尌对导小尐少尒尓尔尕尖尗尘尙尚尛尜尝尞尟尠尡尢尣尤尥尦尧尨尩尪尫尬尭尮尯尰就尲尳尴尵尶尴尸尹尺尻尼尽尾尿"
u"局屁层屃屄居届屇屈屉届屋屌尸屎屏屐屑屒屃屔展屖屗屘屙屚屛屉屝属屟屠屡屡屣层履屦屧屦屩屪屫属屭屮屯屰山屲屳屴屵屶屷屸屹屺屻屼屽屾屿"
u"岀岁岂岃岄岅岆岇岈岉岊岋岌岍岎岏岐岑岒岓岔岕岖岗岘岙岚岛岜岝岞岟岠冈岢岣岤岥岦岧岨岩岪岫岬岭岮岯岰岱岲岳岴岵岶岷岸岹岺岻岼岽岾岿"
u"峀峁峂峃峄峅峆峇峈峉峊峋峌峍峎峏峐峑峒峓峔峕峖峗峘峙峚峛峜峝峞峟峠峡峢峣峤峥峦峧峨峩峪峫峬峭峮峯峰峱峲峳岘峵岛峷峸峹峺峻峼峡峾峿"
u"崀崁崂崃崄崅崆崇崈崉崊崋崌崃崎崏崐昆崒崓崔崕崖岗崘仑崚崛崜崝崞崟崠崡峥崣崤崥崦崧崨崩崪崫岽崭崮崯崰崱崲崳崴崵崶崷崸崹崺崻崼崽崾崿"
u"嵀嵁嵂嵃嵄嵅嵆嵇嵈嵉嵊嵋嵌嵍嵎嵏岚嵑嵒嵓嵔嵕嵖嵗嵘嵙嵚嵛嵜嵝嵞嵟嵠嵡嵢嵣嵤嵥嵦嵧嵨嵩嵪嵫嵬嵭嵮嵯嵰嵱嵲嵳嵴嵵嵶嵷嵸嵹嵺嵻嵼嵽嵾嵿"
u"嶀嵝嶂嶃崭嶅嶆岖嶈嶉嶊嶋嶌嶍嶎嶏嶐嶑嶒嶓嵚嶕嶖崂嶘嶙嶚嶛嶜嶝嶞嶟峤嶡峣嶣嶤嶥嶦峄嶨嶩嶪嶫嶬嶭崄嶯嶰嶱嶲嶳岙嶵嶶嶷嵘嶹岭嶻屿嶽嶾嶿"
u"巀巁巂巃巄巅巆巇巈巉巊岿巌巍巎巏巐巑峦巓巅巕岩巗巘巙巚巛巜川州巟巠巡巢巣巤工左巧巨巩巪巫巬巭差巯巯己已巳巴巵巶巷巸巹巺巻巼巽巾巿"
u"帀币市布帄帅帆帇师帉帊帋希帍帎帏帐帑帒帓帔帕帖帗帘帙帚帛帜帝帞帟帠帡帢帣帤帅带帧帨帩帪师帬席帮帯帰帱帲帐帴帵带帷常帹帺帻帼帽帾帿"
u"帧幁幂帏幄幅幆幇幈幉幊幋幌幍幎幏幐幑幒幓幔幕幖帼帻幙幚幛幜幝幞帜幠幡幢币幤幥幦幧幨幩幪帮帱幭幮幯幰幱干平年幵并幷幸干幺幻幼幽几广"
u"庀庁庂広庄庅庆庇庈庉床庋庌庍庎序庐庑庒库应底庖店庘庙庚庛府庝庞废庠庡庢庣庤庥度座庨庩庪库庬庭庮庯庰庱庲庳庴庵庶康庸庹庺庻庼庽庾庿"
u"廀厕厢廃厩廅廆廇厦廉廊廋廌廍廎廏廐廑廒廓廔廕廖廗廘廙厨廛廜厮廞庙厂庑废广廤廥廦廧廨廪廪廫庐廭廮廯廰廱廲厅廴廵延廷廸廹建回廼廽廾廿"
u"开弁异弃弄弅弆弇弈弉弊弋弌弍弎式弐弑弑弓吊引弖弗弘弙弚弛弜弝弞弟张弡弢弣弤弥弦弧弨弩弪弫弬弭弮弯弰弱弲弪弴张弶强弸弹强弻弼弽弾弿"
u"彀彁彂彃彄彅别彇弹彉彊彋弥彍弯彏彐彑归当彔录彖彗彘汇彚彛彜彝彝彟彠彡形彣彤彦彦彧彨彩彪彫彬彭彮彯彰影彲彳彴彵彶彷彸役彺彻彼彽彾佛"
u"往征徂徃径待徆徇很徉徊律后徍徎徏徐径徒従徔徕徖得徘徙徚徛徜徝从徟徕御徢徣徤徥徦徧徨复循徫彷徭微徯徰徱徲徳徴征徶德徸彻徺徻徼徽徾徿"
u"忀忁忂心忄必忆忇忈忉忊忋忌忍忎忏忐忑忒忓忔忕忖志忘忙忚忛応忝忞忟忠忡忢忣忤忥忦忧忨忩忪快忬忭忮忯忰忱忲忳忴念忶忷忸忹忺忻忼忽忾忿"
u"怀态怂怃怄怅怆怇怈怉怊怋怌怍怎怏怐怑怒怓怔怕怖怗怘怙怚怛怜思怞怟怠怡怢怣怤急怦性怨怩怪怫怬怭怮怯怰怱怲怳怴怵怶怷怸怹怺总怼怽怾怿"
u"恀恁恂恃恄恅恒恇恈恉恊恋恌恍恎恏恐恑恒恓恔恕恖恗恘恙恚恛恜恝恞恟恠恡恢恣恤耻恦恧恨恩恪恫恬恭恮息恰恱恲恳恴恵恶恷恸恹恺恻恼恽恾恿"
u"悀悁悂悃悄悦悆悇悈悉悊悋悌悍悎悏悐悑悒悓悔悕悖悗悘悙悚悛悜悝悮悟悠悡悢患悤悥悦悧您悩悪悫悬悭悮悯悰悱悲德悴怅闷悷悸悹悺悻悼凄悾悿"
u"惀惁惂惃惄情惆惇惈惉惊惋惌惍惎惏惐惑惒惓惔惕惖惗惘惙惚惛惜惝惞惟惠恶惢惣惤惥惦惧惨惩惪惫惬惭惮惯惰恼恽想惴惵惶惷惸惹惺恻惼惽惾惿"
u"愀愁愂愃愄愅愆愇愈愉愊愋愌愍愎意愐愑愒愓愔愕愖愗愘愙愚爱惬愝愞感愠愡愢愣愤愥愦愧悫愩愪愫愬愭愮愯愰愱愲愳怆愵愶恺愸愹愺愻愼愽忾愿"
u"慀慁慂慃栗慅慆殷慈慉慊态慌愠慎慏慐慑慒慓慔慕慖慗惨慙惭慛慜慝慞恸慠慡慢惯悫慥慦慧慨慩怄怂慬慭虑慯慰慱慲悭慴慵庆慷慸慹慺慻戚慽欲慿"
u"憀憁忧憃憄憅憆憇憈憉惫憋憌憍憎憏怜凭愦憓憔憕憖憗憘憙惮憛憜憝憞憟憠憡憢憣愤憥憦憧憨憩憪悯憬憭怃憯憰憱宪憳憴憵忆憷憸憹憺憻憼憽憾憿"
u"懀懁懂勤懄懅懆恳懈应懊懋怿懔懎懏懐懑懒懓懔懕懖懗懘懙懚懛懜懝蒙怼懠懡懢懑懤懥懦懧恹懩懪懫懬懭忧懯懰懱惩懳懴懵懒怀悬懹忏懻惧懽慑懿"
u"恋戁戂戃戄戅戆戆戈戉戊戋戌戍戎戏成我戒戓戋戕或戗战戙戚戛戜戝戞戟戠戡戢戣戤戥戦戗戨戬截戫戬戭戮戯战戯戏戳戴戵户户戸戹戺戻戼戽戾房"
u"所扁扂扃扄扅扆扇扈扉扊手扌才扎扏扐扑扒打扔払扖扗托扙扚扛扜扝扞扟扠扡扢扣扤扥扦执扨扩扪扫扬扭扮扯扰扱扲扳扴扵扶扷扸批扺扻扼扽找承"
u"技抁抂抃抄抅抆抇抈抉把抋抌抍抎抏抐抑抒抓抔投抖抗折抙抚抛抜抝択抟抠抡抢抣护报抦抧抨抩抪披抬抭抮抯抰抱抲抳抴抵抶抷抸抹抺抻押抽抾抿"
u"拀拁拂拃拄担拆拇拈拉拊抛拌拍拎拏拐拑拒拓拔拕拖拗拘拙拚招拜拝拞拟拠拡拢拣拤拥拦拧拨择拪拫括拭拮拯拰拱拲拳拴拵拶拷拸拹拺拻拼拽拾拿"
u"挀持挂挃挄挅挆指挈按挊挋挌挍挎挏挐挑挒挓挔挕挖挗挘挙挚挛挜挝挞挟挠挡挢挣挤挥挦挧挨捝挪挫挬挭挮振挰挱挲挳挴挵挶挷挸挹挺挻挼挽挟挿"
u"捀捁捂捃捄捅捆捇捈捉捊捋捌捍捎捏捐捑捒捓捔捕捖捗捘捙捚捛捜捝捞损捠捡换捣捤捥捦捧舍捩捪扪捬捭据捯捰捱卷捳捴捵捶捷捸捹捺捻捼捽捾捿"
u"掀掁掂扫抡掅掆掇授掉掊掋掌掍掎掏掐掑排掓掔掕掖挜掘挣掚挂掜掝掞掟掠采探掣掤接掦控推掩措掫掬掭掮掯掰掱掲掳掴掵掶掷掸掹掺掻掼掽掾掿"
u"拣揁揂揃揄揅揆揇揈揉揊揋揌揍揎描提揑插揓揔揕揖揗揘揙扬换揜揝揞揟揠握揢揣揤揥揦揧揨揩揪揫揬揭挥揯揰揱揲揳援揵揶揷揸揹揺揻揼揽揾揿"
u"搀搁搂搃搄搅构搇搈搉搊搋搌损搎搏搐搑搒搓搔搕摇捣搘搙搚搛搜搝搞搟搠搡搢搣搤捶搦扇拓搩搪搫搬搭搮搯搰搱搲搳搴揾抢搷搸搹携搻搼搽榨搿"
u"捂摁摂摃摄摅摆摇摈摉摊摋摌摍摎摏摐掴摒摓摔摕摖摗摘摙摚摛掼摝摞搂摠摡摢摣摤摥摦摧摨摩摪摫摬摭摮挚摰摱摲抠摴摵抟摷摸摹折掺摼摽摾摿"
u"撀撁撂撃撄撅撆撇捞撉撊撋撌撍撎挦撑撑撒挠撔撕撖撗撘撙捻撛撜㧑撞挢撠撡掸掸撤拨撦撧撨撩撪抚撬播撮撯撰撱扑揿撴撵撶撷撸撹撺挞撼撽挝捡"
u"擀拥擂擃掳擅擆择擈擉击挡擌操擎擏擐擑擒㧟担擕擖擗擘擙据擛擜擝擞擟挤擡擢捣擤擥擦擧擨擩擪擫拟擭擮摈拧搁掷擳扩擵擶撷擸擹摆擞撸擽扰擿"
u"攀攁攂攃摅攅撵攇攈攉攊攋攌攍攎拢攐攑攒攓拦攕撄攗攘搀攚撺携摄攞攟攠攡攒挛摊攥攦攧攨攩搅攫揽攭攮支攰攱攲攳攴攵收攷攸改攺攻攼攽放政"
u"敀敁敂敃敄故敆敇效敉敊敋敌敍敎敏敐救敒敓敔敕敖败叙教敚敛敜敝敞敟敠敡敢散敤敥敦敧敨敩敪敫敬敭敮敯数敱敲敳整敌敶敷数敹敺敻敼敽敾敿"
u"斀斁敛毙斄斅斆文斈斉斊斋斌斍斎斏斐斑斒斓斔斓斖斗斘料斚斛斜斝斞斟斠斡斢斣斤斥斦斧斨斩斪斫斩断斮斯新斱斲斳斴斵斶断斸方斺斻于施斾斿"
u"旀旁旂旃旄旅旆旇旈旉旊旋旌旍旎族旐旑旒旓旔旕旖旗旘旙旚旛旜旝旞旟无旡既旣旤日旦旧旨早旪旫旬旭旮旯旰旱旲旳旴旵时旷旸旹旺旻旼旽旾旿"
u"昀昁昂昃昄昅昆升昈昉昊昋昌昍明昏昐昑昒易昔昕昖昗昘昙昚昛昜昝昞星映昡昢昣昤春昦昧昨昩昪昫昬昭昮是昰昱昲昳昴昵昶昷昸昹昺昻昼昽显昿"
u"晀晁时晃晄晅晆晇晈晋晊晋晌晍晎晏晐晑晒晓晔晕晖晗晘晙晚晛晜昼晞晟晠晡晢晣晤晥晦晧晨晩晪晫晬晭普景晰晱晲晳晴晵晶晷晸晹智晻晼晽晾晿"
u"暀暁暂暃暄暅暆暇晕晖暊暋暌暍暎暏暐暑暒暓暔暕暖暗旸暙暚暛暜暝暞暟暠暡畅暣暤暥暦暧暨暩暪暂暬暭暮暯暰昵暲暳暴暵暶暷暸暹暺暻暼暽暾暿"
u"曀曁曂曃晔曅历昙曈晓曊曋曌曍曎向曐曑曒曓曔曕暧曗曘曙曚曛曜曝曞曟旷曡曢曣曤曥曦曧昽曩曪曫晒曭曮曯曰曱曲曳更曵曶曷书曹曺曻曼曽曾替"
u"最朁朂会朄朅朆朇月有朊朋朌服朎朏朐朑朒朓朔朕朖朗朘朙朚望朜朝朞期朠朡朢朣朤朥朦胧木朩未末本札朮术朰朱朲朳朴朵朶朷朸朹机朻朼朽朾朿"
u"杀杁杂权杄杅杆杇杈杉杊杋杌杍李杏材村杒杓杔杕杖杗杘杙杚杛杜杝杞束杠条杢杣杤来杦杧杨杩杪杫杬杭杮杯杰东杲杳杴杵杶杷杸杹杺杻杼杽松板"
u"枀极枂枃构枅枆枇枈枉枊枋枌枍枎枏析枑丫枓枔枕枖林枘枙枚枛果枝枞枟枠枡枢枣枤枥枦枧枨枩枪枫枬枭枮枯枰枱枲枳枴枵架枷枸枹枺枻枼枽枾枿"
u"柀柁柂柃柄柅柆柇柈柉柊柋柌柍柎柏某柑柒染柔柕柖柗柘柙柚柛柜柝柞柟柠柡柢柣柤查柦柧柨柩柪柫柬柭柮柯柰柱柲柳柴栅柶柷柸柹柺査柼柽柾柿"
u"栀栁栂栃栄栅栆标栈栉栊栋栌栍栎栏栐树栒栓栔栕栖栗栘栙栚栛栜栝栞栟栠校栢栣栤栥栦栧栨栩株栫栬栭栮栯栰栱栲栳栴栵栶样核根栺栻格栽栾栿"
u"桀桁桂桃桄桅框桇案桉桊桋桌桍桎桏桐桑桒桓桔桕桖桗桘桙桚桛桜桝桞桟桠桡桢档桤桥桦桧桨桩桪桫桬桭桮桯桰桱桲桳桴桵桶桷桸桹桺桻桼桽桾杆"
u"梀梁梂梃梄梅梆梇梈梉梊梋梌梍梎梏梐梑梒梓栀梕梖梗枧梙梚梛梜条梞枭梠梡梢梣梤梥梦梧梨梩梪梫梬梭梮梯械梱棁梳梴梵梶梷梸梹梺梻梼梽梾梿"
u"检棁棂棃弃棅棆棇棈棉棊棋棌棍棎棏棐棑棒棓棔棕枨枣棘棙棚棛棜棝棞栋棠棡棢棣棤棥棦栈棨棩棪棫棬棭森棯棰棱栖棳棴棵梾棷棸棹棺棻棼棽棾棿"
u"椀椁椂椃椄椅椆椇椈椉椊椋椌植椎桠椐椑椒椓椔椕椖椗椘椙椚椛検椝椞椟椠椡椢椣椤椥椦椧椨椩椪椫椬椭椮椯椰椱椲椳椴椵椶椷椸椹椺椻椼椽椾椿"
u"楀楁楂楃楄楅楆楇楈楉杨楋楌楍楎楏楐楑楒枫楔楕楖楗楘楙楚楛楜楝楞楟楠楡楢楣楤楥楦楧桢楩楪楫楬业楮楯楰楱楲楳楴极楶楷楸楹楺楻楼楽楾楿"
u"榀榁概榃榄榅榆榇榈榉榊榋榌榍榎榏榐榑榒榓榔榕谷榗榘榙榚榛榜榝榞榟榠榡榢榣榤榥榦榧榨榩杩榫榬榭荣榯榰榱榅榳榴榵榶榷榸榹榺榻榼榽榾桤"
u"槀槁槂槃槄槅槆槇槈槉槊构槌枪槎槏槐槑槒杠槔槕橐槗様槙槚槛槜槝槞槟槠槡槢槣梿槥槦椠椁槩槪槫槬槭槮槯槰槱槲桨槴槵槶槷槸槹槺槻槼槽槾槿"
u"樀桩乐樃樄枞樆樇樈樉樊樋樌樍樎樏樐梁樒楼樔樕樖樗樘标樚樛樜樝枢樟樠模樢样樤樥樦樧樨権横樫樬樭樮樯樰樱樲樳樴樵樶樷朴树桦樻樼樽樾樿"
u"橀橁橂橃橄橅橆橇桡橉橊桥橌橍橎橏橐橑橒橓橔橕橖橗橘橙橚橛橜橝橞机橠橡椭橣橤橥橦橧橨橩橪横橬橭橮橯橰橱橲橳橴橵橶橷橸橹橺橻橼橽橾橿"
u"檀檩檂檃檄檅檆檇檈柽檊檋檌檍檎檏檐檑檒檓档檕檖檗檘檙檚檛桧楫檞槚檠檡检樯檤檥檦檧檨檩檪檫檬檭梼台檰檱檲槟檴檵檶檷柠檹檺槛檼檽檾檿"
u"櫀櫁櫂柜櫄櫅櫆櫇櫈櫉櫊櫋櫌櫍櫎櫏櫐櫑櫒橹櫔櫕櫖櫗櫘櫙榈栉櫜椟橼栎櫠櫡櫢櫣櫤橱櫦槠栌櫩枥橥榇櫭櫮櫯櫰蘖櫲栊櫴櫵櫶櫷榉櫹棂樱櫼櫽櫾櫿"
u"欀欁欂欃栏欅欆欇欈欉权欋欌欍欎椤欐欑栾欓欔欕榄欗欘欙欚欛欜欝棂欟欠次欢欣欤欥欦欧欨欩欪欫欬欭欮欯欰欱欲欳欴款欶欷欸欹欺欻欼钦款欿"
u"歀歁歂歃歄歅歆歇歈歉歊歋歌歍叹歏欧歑歒歓歔歕歖歗歘歙歚敛歜歝歞欤歠欢止正此步武歧歨歩歪歫歬歭歮歯歰歱岁歳歴歵歶历归歹歺死歼歽歾殁"
u"殀殁殂殃殄殅殆殇殈殉殊残殌殍殎殏殐殑殒殓殔殕殖殗残殙殚殛殜殝殒殟殠殡殢殣殇殥殦殧㱮殩殪殚殬殭殓殡㱩殱歼殳殴段殶殷殸殹杀殻壳殽殾殿"
u"毁毁毂毃毄毅殴毇毈毉毊毋毌母毎每毐毑毒毓比毕毖毗毘毙毚毛毜毝毞毟毠毡毢毣毤毥毦毧毨毩毪毫球毭毮毯毰毱毲毳毴毵毶毷毸毹毺毻毼毽毾毵"
u"氀氁牦氃氄氅氆氇毡氉氊氋氇氍氎氏氐民氒氓气氕氖気氘氙氚氛氜氝氞氟氠氡氢气氤氥氦氧氨氩氪氢氩氭氮氯氰氱氲氲水氵氶氷永凼氺氻氼氽泛氿"
u"汀汁求汃汄汅汆汇汈汉汊汋汌汍泛汏汐汑汒汓汔汕汖汗汘污汚汛汜汝汞江池污汢汣汤汥汦汧汨汩汪汫汬汭汮汯汰汱汲汳汴汵汶汷汸汹决汻汼汽汾汿"
u"沀沁沂沃沄沅沆沇沈沉沊沋沌冱沎沏沐沑没沓沔沕冲沗沘沙沚沛沜沝沞沟沠没沢沣沤沥沦沧沨沩沪沫沬沭沮沯沰沱沲河沴沵沶沷沸油沺治沼沽沾沿"
u"泀况泂泃泄泅泆泇泈泉泊泋泌泍泎泏泐泑泒泓泔法泖泗泘泙泚泛泜泝泞泟泠泡波泣泤泥泦泧注泩泪泫泬泭泮泯泰泱泲泳泴泵泶泷泸泹泺泻泼泽泾泿"
u"洀洁洂洃洄洅洆洇洈洉洊洋洌洍洎洏洐洑洒洓洔洕洖洗洘洙洚洛洜洝洞洟洠洡洢洣洤津洦洧洨泄洪洫洬洭洮洯洰洱洲洳洴洵汹洷洸洹洺活洼洽派洿"
u"浀流浂浃浄浅浆浇浈浉浊测浌浍济浏浐浑浒浓浔浕浖浗浘浙浚浛浜浝浞浟浠浡浢浣浤浥浦浧浨浩浪浫浬浭浮浯浰浱浲浳浴浵浶海浸浃浺浻浼浽浾浿"
u"涀涁涂涃涄涅涆泾消涉涊涋涌涍涎涏涐涑涒涓涔涕涖涗涘涙涚涛涜涝涞涟涠涡涢涣涤涥润涧涨涩涪涫涬涭涮涯涰涱液涳涴涵涶涷涸涹涺涻凉涽涾涿"
u"淀淁淂淃淄淅淆淇淈淉淊淋淌淍淎淏淐淑凄淓淔淕淖淗淘淙泪淛淜淝淞淟淠淡淢淣淤渌淦淧净淩沦淫淬淭淮淯淰深淲淳淴渊涞混淸淹浅添淼淽淾淿"
u"渀渁渂渃渄清渆渇済渉渊渋渌渍渎渏渐渑渒渓渔渕渖渗渘涣渚减渜渝渞渟渠渡渢渣渤渥涡渧渨温渪渫测渭渮港渰渱渲渳渴渵渶渷游渹渺渻渼渽浑渿"
u"湀湁湂湃湄湅湆湇湈湉凑湋湌湍湎湏湐湑湒湓湔湕湖湗湘湙湚湛湜湝浈湟湠湡湢湣湤湥湦涌湨湩湪湫湬湭湮汤湰湱湲湳湴湵湶湷湸湹湺湻湼湽湾湿"
u"満溁溂溃溄溅溆溇沩溉溊溋溌溍溎溏源溑溒溓溔溕准溗溘溙溚溛溜沟溞溟溠溡溢溣溤溥溦溧溨溩溪温溬溭溮溯溰溱溲溳溴溵溶溷溸溹溺溻湿溽溾溿"
u"滀滁滂滃沧灭滆滇滈滉滊滋涤滍荥滏滐滑滒滓滔滕滖滗滘滙滚滛滜滝滞滟滠满滢滣滤滥滦滧滨滩滪滫沪滭滮滞滰滱渗滳滴滵滶卤浒滹滺浐滼滽滚满"
u"漀渔漂漃漄漅漆漇漈漉漊漋漌漍漎漏漐漑漒漓演漕漖漗漘漙沤漛漜漝漞漟漠漡汉涟漤漥漦漧漨漩漪漫渍漭漮漯漰漱涨漳漴溆漶漷渐漹漺漻漼漽漾浆"
u"潀颍潂潃潄潅潆潇潈潉潊潋潌潍潎潏潐泼潒潓洁潕潖潗潘沩潚潜潜潝潞潟潠潡潢潣润潥潦潧潨潩潪潫潬潭潮浔溃潱潲潳潴潵潶滗潸潹潺潻潼潽潾涠"
u"涩澁澂澃澄澅浇涝澈澉澊澋澌澍澎澏澐澑澒澓澔澕澖涧澘澙澚澛澜澝澞澟渑澡澢澣泽澥滪澧澨泶澪澫澬澭浍澯澰淀澲澳澴澵澶澷澸澹澺澻澼澽澾澿"
u"激浊濂浓濄濅濆濇濈濉濊濋濌濍濎濏濐濑濒濓濔湿濖濗泞濙濚濛濜濝濞济濠濡濢濣涛濥濦濧濨濩濪滥浚濭濮濯潍滨濲濳濴濵濶濷濸濹溅濻泺濽滤濿"
u"瀀瀁瀂瀃瀄滢渎㲿瀈泻瀊沈瀌瀍瀎浏瀐瀑瀒瀓瀔濒瀖瀗泸瀙瀚瀛瀜沥瀞潇潆瀡瀢瀣瀤瀥潴泷濑瀩瀪瀫瀬瀭瀮瀯弥瀱潋瀳瀴瀵瀶瀷瀸瀹瀺瀻瀼瀽澜瀿"
u"灀灁灂沣滠灅灆灇灈灉灊灋灌灍灎灏灐洒灒灓灔漓灖灗滩灙灚灛灜灏灞灟漤灡灢湾滦灥灦滟灨灩灪火灬灭灮灯灰灱灲灳灴灵灶灷灸灹灺灻灼灾灾灿"
u"炀炁炂炃炄炅炆炇炈炉炊炋炌炍炎炏炐炑炒炓炔炕炖炗炘炙炚炛炜炝炞炟炠炡炢炣炤炥炦炧炨炩炪炫炬炭炮炯炰炱炲炳炴炵炶炷炸点为炻炼炽炾炿"
u"烀烁烂烃烄烅烆烇烈烉烊烋烌烍烎乌烐烑烒烓烔烕烖烗烘烙烚烛烜烝烞烟烠烡烢烣烤烥烦烧烨烩烪烫烬热烮烯烰烱烲烳烃烵烶烷烸烹烺烻烼烽烾烿"
u"焀焁焂焃焄焅焆焇焈焉焊焋焌焍焎焏焐焑焒焓焔焕焖焗焘焙焚焛焜焝焞焟焠无焢焣焤焥焦焧焨焩焪焫焬焭焮焯焰焱焲焳焴焵然焷焸焹焺焻焼焽焾焿"
u"煀煁煂煃煄煅煆煇煈炼煊煋煌煍煎煏煐煑炜煓煔煕煖煗煘烟煚煛煜煝煞煟煠煡茕煣煤焕煦照煨烦煪煫炀煭煮煯煰㶽煲煳煴煵煶煷煸煹煺煻煼煽煾煿"
u"熀熁熂熃熄煴熆熇熈熉熊熋熌熍熎熏熐熑荧熓熔熕熖炝熘熙熚熛熜熝熞熟熠熡熢熣熤熥熦熧熨熩熪熫熬熭熮熯熰热颎熳熴熵熶熷熸熹熺熻熼熽炽熿"
u"燀烨燂燃焰燅燆燇灯炖燊燋燌燍燎燏磷燑烧燓燔燕燖燗燘烫燚燛焖燝燞营燠燡燢燣燤燥灿燧燨燩燪燫毁烛燮燯燰燱燲燳烩燵㶶燷燸燹燺熏烬燽焘耀"
u"爀爁爂爃爄爅爆爇爈爉爊爋爌烁爎爏炉爑爒爓爔爕爖爗爘爙爚烂爜爝爞爟爠爡爢爣爤爥爦爧爨爩爪爫爬争爮爯爰爱为爳爴爵父爷爸爹爷爻爼爽尔爿"
u"床牁牂牃牄牅墙片版牉牊笺牌牍牎牏牐牑牒牓牔牕牖牗牍牙牚牛牜牝牞牟牠牡牢牣牤牥牦牧牨物牪牫牬牭牮牯牰牱牲牳牴牵牶牷牸特牺牻牼牵牾牿"
u"犀犁犂犃犄犅犆犇犈犉犊犋犌犍犎犏犐犑犒犓犔犕荦犗犘犙犚犛犜犝犞犟犠犡犊犣犤犥犦牺犨犩犪犫犬犭犮犯犰犱犲犳犴犵状犷犸犹犺犻犼犽犾犿"
u"状狁狂狃狄狅狆狇狈狉狊狋狌狍狎狏狐狑狒狓狔狕狖狗狘狙狚狛狜狝狞狟狠狡狢狣狤狥狦狧狨狩狪狫独狭狮狯狰狱狲狳狴狵狶狷狸狭狺狻狼狈狾狿"
u"猀猁猂猃猄猅猆猇猈猉猊猋猌猍猎猏猐猑猒猓猔猕猖猗猘狰猚猛猜猝猞猟猠猡猢猣猤猥猦猧猨猩猪猫猬猭献猯猰猱猲猳猴猵犹猷猸猹猺狲猼猽猾猿"
u"獀犸獂呆狱狮獆獇獈獉獊獋獌獍奖獏獐獑獒獓獔獕獖獗獘獙獚獛獜獝獞獟獠獡獢獣獤獥獦獧独獩狯猃獬獭狝獯狞㺍获獳獴猎獶犷兽獹獭献猕獽獾獿"
u"猡玁玂玃玄玅玆率玈玉玊王玌玍玎玏玐玑玒玓玔玕玖玗玘玙玚玛玜玝玞玟玠玡玢玣玤玥玦玧玨玩玪玫玬玭玮环现玱玲玳玴玵玶玷玸玹玺玻玼玽玾玿"
u"珀珁珂珃珄珅珆珇珈珉珊珋珌珍珎珏珐珑珒珓珔珕珖珗珘珙珚珛珜珝珞珟珠珡珢珣珤珥珦珧珨珩珪珫珬班珮珯珰珱珲珳珴珵珶珷珸珹珺珻珼珽现珿"
u"琀琁琂球琄琅理琇琈琉琊琋琌琍琎琏琐琑琒琓琔琕琖琗琘琙琚琛琜琝琞琟琠琡琢琣琤琥琦琧琨琩琪琫琬琭琮琯琰琱琲琳琴琵琶琷琸琹珐琻琼琽琾珲"
u"瑀瑁瑂瑃瑄瑅瑆瑇瑈瑉瑊玮瑌瑍瑎瑏瑐瑑玚瑓瑔瑕瑖瑗瑘瑙瑚瑛瑜瑝瑞瑟瑠瑡瑢琐瑶瑥瑦瑧瑨莹玛瑫瑬瑭瑮琅瑰瑱玱瑳瑴瑵瑶瑷瑸瑹瑺瑻瑼瑽瑾瑿"
u"璀璁璂璃璄璅璆璇璈琏璊璋璌璍璎璏璐璑璒璓璔璕璖璗璘璙璚璛璜璝璞璟璠璡璢玑璤璥瑷璧璨璩璪珰璬璭璮璯环璱璲璳璴璵璶璷璸璹璺璻璼玺璾璿"
u"瓀瓁瓂瓃瓄瓅瓆瓇瓈瓉琼瓋瓌瓍瓎珑瓐瓑瓒瓓璎瓕瓖瓗瓘瓙瓒瓛瓜瓝瓞瓟瓠瓡瓢瓣瓤瓥瓦瓧瓨瓩瓪瓫瓬瓭瓮瓯瓰瓱瓲瓳瓴瓵瓶瓷瓸瓹瓺瓻瓼瓽瓾瓿"
u"甀甁甂甃甄甅甆甇甈甉甊甋瓯甍甎甏甐甑甒甓甔瓮甖甗甘甙甚甛甜甝甞生甠甡产产甤甥甦甧用甩甪甫甬甭甮甯田由甲申甴电甶男甸甹町画甼甽甾甿"
u"畀畁畂畃畄畅畆畇畈畉畊畋界畍畎畏畐畑畒畓畔畕畖畗畘留畚畛畜亩畞畟畠畡毕畣畤略畦畧畨畩番画畬畭畮畯异畱畲畳畴畵当畷畸畹畺畻畼畽畾畿"
u"疀疁疂疃疄疅疆畴疈疉叠疋疌疍疎疏疐疑疒疓疔疕疖疗疘疙疚疛疜疝疞疟疠疡疢疣疤疥疦疧疨疩疪疫疬疭疮疯疰疱疲疳疴疵疶疷疸疹疺疻疼疽疾疿"
u"佝痁痂痃痄病痆症痈痉痊痋痌痍痎痏痐痑痒痓痔痕痖痗痘痉痚痛痜痝痞痟酸痡痢痣痤痥痦痧痨痩痪痫痬痭痮痯痰痱痲痳痴痵痶痷痸痹痺痻痼痽疴痿"
u"瘀瘁痖瘃瘄瘅瘆瘇瘈瘉瘊疯瘌疡瘎瘏瘐瘑瘒痪瘔瘕瘖瘗瘘瘙瘚瘛瘜瘝瘗瘟瘠疮瘢瘣瘤瘥瘦疟瘨瘩瘪瘫瘬瘭瘆瘯瘰瘱疭瘳瘴瘵瘶瘷瘸瘹瘘瘘瘼瘽瘾瘿"
u"癀癁疗癃癄癅痨痫癈瘅癊癋癌癍癎癏癐癑愈癓癔癕癖癗疠癙癚癛癜癝癞瘪癠痴痒癣疖症癦疬癨癞癪癫癣瘿瘾癯痈瘫癫癳癴癵癶癷癸癹発登发白百癿"
u"皀皂皂皃的皅皆皇皈皉皊皋皌皍皎皏皐皑皒皓皔皕皖皗皘皙皑皛皜皝皞皟皠皡皢皣皤皥皦皧皨皩皪皫皬皭皮皯疱皱皲皳皴皵皶皷皲皹皱皻皼皽皾皿"
u"盀盁盂杯盄盅盆盇盈盉益盋盌盍盎盏盐监盒盓盔盕盖盗盘盙盚盛盗盝盏盟盠尽盢监盘盥盦卢盨盩荡盫盬盭目盯盰盱盲盳直盵盶盷相盹盺盻盼盽盾盿"
u"眀省眂眃眄眅眆眇眈眉眊看県眍眎眏眐眑眒眓眔眕眖眗眘眙眚眛眜眝眞真眠眡眢眣眤眦眦眧眨眩眪眫眬眭眮眯眰眱眲眳眴眵眶眷眸眹眺眻眼眽众眿"
u"着睁睂睃睄睅睆睇睈睉睊睋睌睍睎困睐睑睒睓睔睕睖睗睘睙睚睛睁睝睐睟睠睡睢督睤睥睦睧睨睩睾睫睬睭睮睯睰睱睲睳睴睵睶睷睸睹睺睻睼睽睾睿"
u"瞀瞁瞂瞃瞄瞅瞆眯瞈瞉瞊瞋瞌瞍瞎瞏瞐瞑瞒瞓瞔瞕瞖瞗眍瞙瞚瞛䁖瞝瞒瞟瞠瞡瞢瞣瞤瞥瞦瞧瞨瞩瞪瞫瞬了瞮瞯瞰瞱瞲瞳瞴瞵瞆瞷瞸瞹瞺瞻睑瞽瞾瞿"
u"矀矁矂矃矄矅矆矇矈矉矊矋矌矍矎矏矐矑矒眬矔矕矖矗矘矙瞩矛矜矝矞矟矠矡矢矣矤知矦矧矨矩矪矫矬短矮矫矰矱矲石矴矵矶矷矸矹矺矻矼矽矾矿"
u"砀码砂砃砄砅砆砇砈砉砊砋砌砍砎砏砐砑砒砓研砕砖砗砘砙砚砛砜砝砞砟砠砡砢砣砤砥砦砧砨砩砪砫砬砭砮砯砰砱炮砳破砵砶砷砸砹砺砻砼砽砾砿"
u"础硁硂硃硄硅硆硇硈硉硊硋硌硍硎硏硐硑硒硓硔硕硖硗硘硙硚硛硁硝硞硟硠硡硢硣硖硥硦硧砗硩硪硫硬硭确砚硰硱硲硳硴硵硶硷硸硹硺硻硼硽硾硿"
u"碀碁碂碃碄碅碆碇碈碉碊碋碌碍碎碏碐碑碒碓碔碕碖碗碘碙碚碛碜碝碞碟碠碡碢碣碤碥碦碧碨硕碪碫碬砀碮碯碰碱碲碳碴碵碶碷砜碹确碻码碽碾碿"
u"磀磁磂磃磄磅磆磇磈磉磊磋磌磍磎磏磐硙磒磓磔磕磖磗磘磙砖磛磜磝磞磟磠磡磢碜磤磥磦碛磨磩磪磫磬磭磮矶磰磱磲磳磴磵磶磷磸磹磺磻磼硗磾磿"
u"礀礁礂礃礄礅硷礇礈礉礊礋礌礍础礏礐礑礒礓礔礕礖礗礘碍礚礛礜礝礞礟礠礴礢礣礤礥矿礧礨礩砺砾矾礭炮礯礰砻礲礳礴礵礶礷礸礹示礻礼礽社礿"
u"祀祁祂祃祄祅祆祇祈祉祊祋祌祍祎祏祐祑祒祓祔秘祖祗祘祙祚祛祜祝神祟祠祡祢祣祤祥祦祧票祩祪祫祬祭祮祯祰祱祲祳祴祵祶祷祸祹祺祻祼祽祾禄"
u"禀禁禂禃禄禅禆禇禈禉禊禋禌祸祯福禐禑禒禓禔祎禖禗禘禙禚禛禜禝禞禟禠祃禢禣禤禥御禧禨禩禅禫禬禭礼禯祢祷禲禳禴禵禶禷禸禹禺离禼禽禾秃"
u"秀私秂秃秄秅秆秇籼秉秊秋秌种秎秏秐科秒秓秔秕秖秗秘秙秚秛秜秝秞租秠秡秢秣秤秥秦秧秨秩秪秫秬秭秮积称秱秲秳秴秵秶秷秸秹秺移秼秽秾秿"
u"稀稁稂稃稄税稆稇秆稉稊程稌稍税䅉稐稑稒稓稔稕稖稗稘稙稚稛棱稝稞禀稠稡稢稣稤稥稦稧稨稩稪稫稬稭种稯稰称稲稳稴稵稶稷稸稹稺稻稼稽稾稿"
u"谷穁穂穃穄穅穆穇穈穉穊穋稣积颖穏穐穑穒穓穔穕穖穗穘穙穚穛穜穝穞穟秾穑秽穣穤穥穦穧穨稳穪获穬稆穮穯穰穱穲穳穴穵究穷穸穹空穻穼穽穾穿"
u"窀突窂窃窄窅窆窇窈窉窊窋窌窍窎窏窐窑窒窓窔窕窖窗窘窙窚窛窜窝窞窟窠窡窢窣窤窥窦窧窨窝洼窫窬窭穷窑窰窱窲窳窴窎窭窷窸窹窥窻窼窽窾窿"
u"竀竁竂竃窜窍竆窦灶竉窃立竌竍竎竏竐竑竒竓竔竕竖竗竘站竚竛竜竝竞竟章竡竢竣竤童竦竧竨竩竖竫竬竭竮端竰竱竲竳竴竵竞竷竸竹竺竻竼竽竾竿"
u"笀笁笂笃笄笅笆笇笈笉笊笋笌笍笎笏笐笑笒笓笔笕笖笗笘笙笚笛笜笝笞笟笠笡笢笣笤笥符笧笨笩笪笫第笭笮笯笰笱笲笳笴笵笶笷笸笹笺笻笼笽笾笿"
u"筀筁筂筃筄筅笔筇筈等筊筋筌笋筎筏筐筑筒筓答筕策筗筘筙筚筛筜筝筞筟筠筡筢筣筤筥筦笕筨筩筪筫筬筭筮筯筰筱筲筳䇲筵筶筷筸筹筺筻筼筽签筿"
u"简箁箂箃箄箅箆个箈箉箊笺箌箍篪筝箐箑箒箓箔箕箖算箘箙箚箛箜钳箞箟箠管箢箣箤箥箦箧箨箩箪箫箬箭箮箯箰箱箲箳箴箵箶箷箸箹箺箻箼箽箾箿"
u"节篁篂篃范篅篆篇篈筑篊箧篌篍篎篏篐篑篒篓筼篕篖篗篘篙篚篛篜篝篞篟篠篡篢篣笃篥篦篧篨筛篪篫篬篭篮篯篰篱篲筚篴篵篶篷篸篹篺篻篼篽篾篿"
u"箦簁簂簃簄簅筘簇簈簉簊簋簌篓簎簏簐簑簒簓簔簕簖簗簘簙簚簛簜簝箪簟簠简簢篑簤簥簦簧簨簩簪箫簬簭簮簯簰簱簲簳簴簵簶檐簸筜簺簻簼签帘簿"
u"籀籁籂篮籄籅籆籇籈籉籊籋筹籍籎籏藤籑籒籓籔籕籖籗籘箓籚籛箨籝籞籁笼籡籢籣签籥籦籧籨笾簖籫篱籭箩籯籰籱吁米籴籵籶籷籸籹籺类籼籽籾籿"
u"粀粁粂粃粄粅粆粇粈粉粊粋粌粍粎粏粐粑粒粓粔粕粖粗粘粙粚粛粜粝粞粟粠粡粢粣粤粥粦妆粨粩粪粫粬粭粮粯粰粱粲粳粴粤粶粷粸粹粺粻粼粽精粿"
u"糀糁糂糃糄糅糆糇糈糉糊糋糌糍糎糏糐糑糒糓糔糕糖糗糘糙糚糛糜糁粪糟糠糡糢糣糤糥糦粮糨糩糪糫糬糭糮糯团糱粝糳籴糵粜糷糸纟糺系糼糽纠糿"
u"纪紁纣紃约红纡纥纨纫紊纹紌纳紎紏纽紑紒纾纯纰纼纱纮纸级纷纭纴紞紟素纺索紣紤紥紦紧紨紩紪紫䌷紭扎累细绂绁绅紴纻紶紷紸绍绀紻绋紽紾绐"
u"绌絁终弦组䌹绊絇絈絉絊絋経絍绗絏结絑絒絓絔绝絖絗絘絙絚绦絜绔绞絟絠络绚絣絤絥给絧绒絩絪絫絬絭絮絯绖统丝绛絴絵绝絷絸绢絺絻絼絽絾絿"
u"綀绑綂绡綄綅绠綇绨绣綊綋绤綍綎绥䌼捆綒经綔綕綖綗綘継続綛综綝缍綟绿綡绸绻綤綥綦綧綨綩綪线绶维綮绹绾纲网绷缀彩綶綷纶绺绮绽綼绰绫绵"
u"緀緁緂緃绲緅緆缁緈緉紧绯緌緍緎総緐绿绪绬绱緕緖缃缄缂线緛緜缉缎緟缔缗緢缘緤緥缌緧编缓緪緫缅緭緮纬緰缑缈緳练緵缏緷緸缇緺致緼緽緾緿"
u"縀縁縂縃縄縅縆縇萦缙缢缒縌縍縎縏绉缣縒縓縔缊縖缞縘縙縚缚縜缜缟缛縠縡縢县縤縥縦绦縨縩縪缝縬缡缩縯縰纵缧䌸纤缦絷缕縸缥縺縻縼总绩縿"
u"繀繁繂绷繄缫缪繇繈繉繊繋繌繍繎繏繐繑缯繓织缮繖繗繘繙缭繛繜繝绕繟繠绣缋繣繤繥繦繧繨绳绘系繬茧缰缳缲繱繲缴繴繵繶繷䍁绎繺繻继缤缱䍀"
u"纀纁纂纃纄纅纆纇缬纉纩纋续累纎缠纐纑纒缨才纕纤纗缵纙纚纛缆纝纞纟纠纡红纣纤纥约级纨纩纪纫纬纭纮纯纰纱纲纳纴纵纶纷纸纹纺纻纼纽纾线"
u"绀绁绂练组绅细织终绉绊绋绌绍绎经绐绑绒结绔绕绖绗绘给绚绛络绝绞统绠绡绢绣绤绥绦继绨绩绪绫绬续绮绯绰绱绲绳维绵绶绷绸绹绺绻综绽绾绿"
u"缀缁缂缃缄缅缆缇缈缉缊缋缌缍缎缏缐缑缒缓缔缕编缗缘缙缚缛缜缝缞缟缠缡缢缣缤缥缦缧缨缩缪缫缬缭缮缯缰缱缲缳缴缵缶缷缸缹缺缻缼钵缾缿"
u"罀罁罂罃罄罅罆罇坛罉罊罋罂罍坛罏罐网罒罓罔罕罖罗罘罙罚罛罜罝罞罟罠罡罢挂罤罥罦罧罨罩罪罫罬罭置罯罚罱署罳罴骂罶罢罸罹罺罻罼罽罾罿"
u"羀羁羂羃羄罗罴羇羁羉羊芈羌羍美羏羐羑羒羓羔羕羖羗羘羙羚羛羜羝羞羟羠羡羢群群羟羦羧羡义羪羫羬羭羮羯羰羱羲羳羴羵膻羷羸羹羺羻羼羽羾羿"
u"翀翁翂翃翄翅翆翇翈翉翊翋翌翍翎翏翐翑习翓翔翕翖翗翘翙翚翛翜翝翞翟翠翡翢翣翤翥翦翧翨翩翪玩翬翭翮翯翰翱翲翳翴翵翶翷翸翘翱翻翼翽翾翿"
u"耀老耂考耄者耆耇耈耉耊耋而耍耎耏耐耑耒耓耔耕耖耗耘耙耚耛耜耝耞耟耠耡耢耣耤耥耦耧耨耩耪耫耧耭耢耯耰耱耲耳耴耵耶耷耸耹耺耻耼耽耾耿"
u"聀聁聂聃聄聅聆聇聈聉聊聋职聍聎聏聐聑聒聓联聕圣聗聘聙聚聛聜聝闻聟聠聡聢聣聤聥聦聧聨聩聪聫聬聭聮联聪聱声耸聴聩聂职聸聍聺聻聼听聋聿"
u"肀肁肂肃肄肃肆肇肈肉肊肋肌肍肎肏肐肑肒肓肔肕肖肗肘肙肚肛肜肝肞肟肠股肢肣肤肥肦肧肨肩肪肫肬肭肮肯肰肱育肳肴肵肶肷肸肹肺肻肼肽肾肿"
u"胀胁胂胃胄胅胆胇胈胉胊胋背胍胎胏胐胑胒胓胔胕胖胗胘胙胚胛胜胝胞胟胠胡胢胣胤胥胦胧胨胩胪胫胬胭胮胯胰胱胲胳胴胵胶胷胸胹胺胻胼能胾胿"
u"脀脁脂脃脄胁脆脇脉脉脊脋脌脍脎脏脐脑脒脓脔脕脖脗脘脙脚胫脜脝脞脟脠脡脢唇脤脥脦脧脨脩脪脱脬脭脮脯脰脱脲脳脴脵脶脷脸胀脺脻脼脽脾脿"
u"腀腁腂腃腄腅腆腇腈腉腊腋腌腍肾腏腐腑腒腓腔腕胨腗腘腙腚腛腜腝腞腟腠脶腢腣腤腥脑腧腨腩腪肿腬腭腮腯腰腱腲脚腴腵腶腷肠腹腺腻腼腽腾腿"
u"膀膁膂腽膄膅膆膇膈膉膊膋膌膍膎膏膐膑膒膓膔膕膖膗膘膙肤膛膜膝膞膟胶膡膢膣膤膥膦膧膨腻膪膫膬膭膮膯膰膱膲膳膴膵膶膷膸膹膺膻膼胆脍脓"
u"臀臁臂臃臄臅臆臇臈脸臊臋臌脐臎膑臐臑臒臓臔臕臖臗腊臙胪臛臜臝臞脏脔臡臜臣臤卧臦臧临臩自臫臬臭臮臯臰臱臲至致臵臶臷臸臹台臻臼臽臾臿"
u"舀舁舂舃舄舅舆与兴举旧舋舌舍舎舏舐舑舒舓舔舕铺舗舘舙舚舛舜舝舞舟舠舡舢舣舤舥舦舧舨舩航舫般舭舮舯舰舱舲舳舴舵舶舷舸船舺舻舼舽舾舿"
u"艀艁艂艃艄艅艆艇艈艉艊艋艌艍艎艏艐艑艒艓艔艕艖艗艘舱艚艛艜艝艞艟艠艡艢艣舣艥舰艧艨艩艪舻艬艭艮良艰艰色艳艴艵艶艳艸艹艺艻艼艽艾艿"
u"芀芁节芃芄芅芆芇芈芉芊芋芌芍芎芏芐芑芒芓芔芕芖芗芘芙芚芛芜芝芞芟芠芡芢芣芤芥芦芧芨芩芪芫芬芭芮芯芰花芲芳芴芵芶芷芸芹芺刍芼芽芾芿"
u"苀苁苂苃苄苅苆苇苈苉苊苋苌苍苧苏苐苑苒苓苔苕苖苗苘苙苚苛苜苝苞苟苠苡苢苣苤若苦苎苨苩苪苫苬苭苮苯苰英苲苳苴苵苶苷苸苹苺苻苼苽苾苿"
u"茀茁茂范茄茅茆茇茈茉茊茋茌茍茎茏茐茑茒茓茔茕茖茗茘茙茚茛茜茝茞茟茠茡茢茣茤茥茦茧茨茩茪茫茬茭茮茯茰茱兹茳茴茵茶茷茸茹茺茻茼茽茾茿"
u"荀荁荂荃荄荅荆荇荈草荆荋荌荍荎荏荐荑荒荓荔荕荖荗荘荙荚荛荜荝荞荟荠荡荢荣荤荥荦荧荨荩荪荫荬荭荮药荰荱荲豆荴荵荶荷荸荹荺荻荼荽荾荿"
u"莀莁莂莃莄莅莆莇莈莉庄莋莌莍莎莏莐莑莒莓莔莕茎莗莘莙莚莛莜莝莞莟莠莡荚莣莤莥莦苋莨莩莪莫莬莭莮莯莰莱莲莳莴莵莶获莸莹莺莻莼莽莾莿"
u"菀菁菂菃菄菅菆菇菈菉菊菋菌菍菎菏菐菑菒果菔菕菖菗菘菙菚菛菜菝菞菟菠菡菢菣菤菥菦菧菨菩菪菫菬菭菮华菰菱菲菳菴菵菶菷烟菹菺菻菼菽菾菿"
u"萀萁萂萃萄萅萆苌萈萉莱萋萌萍萎萏萐萑萒萓萔萕萖萗萘萙萚萛萜萝萞萟萠萡萢萣萤营萦萧萨萩萪萫万萭萮萯萰萱萲萳萴莴萶萷萸萹萺萻萼落萾萿"
u"葀葁葂葃葄葅葆葇葈叶葊葋葌葍葎葏葐葑荭葓葔葕葖着葘葙葚葛葜葝葞葟葠葡葢董荮葥苇葧葨葩葪葫葬葭葮药葰葱葲葳葴葵葶荤葸葹葺葻葼葽葾葿"
u"蒀蒁蒂蒃蒄蒅蒆蒇蒈蒉蒊蒋蒌蒍蒎蒏搜蒑蒒莼莳蒕蒖蒗蒘蒙蒚蒛蒜蒝莅蒟蒠蒡蒢蒣蒤蒥蒦蒧蒨蒩蒪蒫蒬蒭蒮蒯蒰蒱蒲蒳蒴蒵蒶蒷蒸蒹蒺蒻苍蒽蒾蒿"
u"荪蓁蓂蓃蓄蓅席蓇蓈蓉蓊盖蓌蓍蓎蓏蓐蓑蓒蓓蓔蓕蓖蓗蓘蓙蓚蓛蓜蓝蓞蓟蓠蓡蓢蓣蓤蓥蓦蓧蓨蓩蓪蓫蓬蓭莲苁蓰蓱蓲蓳蓴蓵蓶蓷蓸蓹蓺蓻蓼荜蓾蓿"
u"蔀蔁蔂蔃蔄蔅蔆蔇蔈蔉蔊蔋蔌蔍蔎蔏蔐蔑蔒蔓卜蔕蔖蔗蔘蔙蔚蔛蔜蔝蒌蔟蔠蔡蔢蒋蔤葱茑蔧蔨蔩蔪蔫蔬荫蔮蔯蔰蔱蔲蔳麻蔵蔶蔷蔸蔹蔺蔻蔼蔽蔾蔿"
u"蕀荨蕂蕃蕄蕅蒇蕇蕈蕉蕊蕋蕌蕍荞蕏蕐蕑荬芸蕔莸蕖蕗荛蕙蕚蕛蕜蕝蕞蕟蕠蕡蒉蕣蕤蕥蕦蕧蕨荡芜蕫蕬萧蕮蕯蕰蕱蕲蕳蕴蕵蕶蓣蕸蕹蕺蕻蕼蕽蕾蕿"
u"蕰薁薂薃薄薅薆薇荟薉蓟薋芗薍薎薏薐姜薒薓蔷薕薖薗荙薙薚薛薜薝薞莶薠薡薢薣薤薥荐薧薨萨薪薫薬薭薮薯薰薱薲䓕苧薵薶薷薸薹荠薻薼薽薾薿"
u"藀藁藂藃藄藅藆藇藈借藊藋藌蓝荩藏藐藑藒藓藔藕藖藗藘藙藚藛藜艺藞藟藠藡藢藣藤药藦藧藨藩薮藫藬藭藮藯藰藱藲藳蕴藵苈薯藸蔼蔺藻藼藽藾藿"
u"蘀蘁蘂蘃蕲蘅芦苏蘈蘉蕴苹蘌蘍蘎蘏蘐蘑蘒蘓蘔蘕蘖蘗蘘蘙藓蘛蘜蘝蔹蘟蘠蘡茏蘣蘤蘥蘦蘧蘨蘩蘪蘫蘬兰蘮蘯蘰蘱蘲蘳蘴蘵蘶蘷蘸蘹蓠蘻蘼蘽蘾萝"
u"虀虁虂虃虄虅蔂虇虈虉虊虋虌虍虎虏虐虑虒虓虔处虖虗虘虙虚虚虏虝虞号虠虡虢虣虤虥虦亏虨虩虪虫虬虭虮虬虰虱虲虳虴虵虶虷虸虹虺虻虼虽虾虿"
u"蚀蚁蚂蚃蚄蚅蚆蚇蚈蚉蚊蚋蚌蚍蚎蚏蚐蚑蚒蚓蚔蚕蚖蚗蚘蚙蚚蚛蚜蚝蚞蚟蚠蚡蚢蚣蚤蚥蚦蚧蚨蚩蚪蚫蚬蚭蚮蚯蚰蚱蚲蚳蚴蚵蚶蚷蚸蚹蚺蚻蚼蚽蚾蚿"
u"蛀蛁蛂蛃蛄蛅蛆蛇蛈蛉蛊蛋蛌蛍蛎蛏蛐蛑蛒蛓蛔蛕蛖蛗蛘蛙蛚蛛蛜蛝蛞蛟蛠蛡蛢蛣蛤蛥蛦蛧蛨蛩蛪蛫蛬蛭蛮蛯蛰蛱蛲蛳蛴蛵蛶蛷蛸蛹蛱蜕蛼蛽蛾蛿"
u"蜀蜁蜂蜃蜄蜅蚬蜇蜈蜉蜊蜋蜌蜍蜎蜏蜐蜑蜒蜓蜔蜕蜖蜗蜘蜙蜚蜛蜜蜝蜞蜟蜠蜡蜢蜣蜤蜥蜦蜧蜨蜩蜪蜫蜬蜭蜮蜯蜰蜱蜲蜳蜴蜵蜶蜷蜸蜹蜺蜻蜼蜽蜾蜿"
u"蝀蝁蝂蝃蝄蝅蝆蝇蝈蝉蝊蝋蝌蝍蝎蝏蝐蝑蝒蝓蝔蚀蝖蝗蝘蝙蝚蝛蝜蝝蝞猬蝠蝡蝢蝣蝤蝥虾蝧虱蝩蝪蝫蝬蝭蝮蝯蝰蝱蝲蝳蝴蝵蝶蝷蜗蝹蝺蝻蝼蝽蝾蝿"
u"螀螁螂螃蛳螅螆螇螈螉螊螋螌融螎螏螐螑螒螓螔螕螖螗螘螙螚螛螜螝蚂螟螠螡萤螣螤螥螦螧螨螩螪螫螬螭䗖螯螰螱螲螳螴螵螶螷螸螹螺蝼螼螽螾螀"
u"蟀蟁蟂蟃蛰蟅蟆蟇蝈蟉蟊蟋蟌蟍螨蟏蟐蟑蟒蟓蟔蟕蟖蟗蟘蟙蟚蟛蟜蟝蟞蟟蟠蟡蟢虮蟤蟥蟦蟧蟨蟩蟪蟫蝉蟭蟮蛲蟰蟱虫蟳蟴蟵蛏蟷蟸蟹蟺蚁蟼蟽蟾蟿"
u"蠀蠁蠂蠃蠄蝇虿蠇蠈蠉蠊蠋蠌蝎蠎蠏蛴蝾蠒蠓蚝蠕蠖蠗蠘蠙蠚蠛蠜蠝蠞蜡蠠蠡蠢蛎蠤蠥蠦蠹蟏蠩蠪蠫蠬蠭蠮蠯蠰蛊蠲蠳蠴蠵蚕蠷蠸蠹蠺蛮蠼蠽蠾蠿"
u"血衁衂衃衄衅众衇衈衉蔑衋行衍衎衏衐衑衒术衔衕衖街衘衙胡卫衜冲衞衟衠衡衢衣衤补衦衧表衩衪衫衬衭衮衯衰衱衲衳衴衵衶衷衸衹衺衻衼衽衾衿"
u"袀袁袂袃袄袅袆袇袈袉袊袋袌袍袎袏袐袑袒袓袔袕袖袗袘袙袚袛袜袝衮袟袠袡袢袣袤袥袦袧袨袩袪被袬袭袮袯袰袱袲袳绔袵袶袷袸袹袺袻袼袽袾袿"
u"裀裁裂裃裄装裆裇裈裉袅裋裌裍裎里裐裑裒裓裔裕裖裗裘裙裚裛补装裞裟裠里裢裣裤裥裦裧裨裩裪裫裬裭裮裯裰裱裲裳裴裵裶裷裸裹裺裻裼制裾裿"
u"褀褁褂褃褄褅褆复褈褉褊褋裈褍褎褏褐褑褒褓褔褕褖褗袆褙褚褛褜褝褞褟褠褡褢褣褤褥褦褧褨褩褪褫褬褭褮褯褰褱裤裢褴褵褶褷褛褹褺亵褼褽褾褿"
u"襀襁襂襃襄襅襆裥襈襉襊襋襌襍襎袯襐襑襒襓襔襕袄襗襘襙襚襛襜裣襞襟裆襡襢襣褴襥襦襧襨襩袜襫䙓襭襮衬襰襱袭襳襴襵襶襷襸襹襺襻襼襽襾西"
u"覀要覂覃覄覅覆覇核覉覊见覌覍觃规覐覑覒觅覔覕视覗觇覙覚覛覜覝覞覟覠觋覢覣覤觍觎覧覨覩亲覫觊覭覮觏覰覱觐観覴覵覶觑覸覹觉覻覼览覾觌"
u"观见观觃规觅视觇览觉觊觋觌觍觎觏觐觑角觓觔觕觖觗觘觙觚觛觜觝觞觟觠觡觢解觤觥触觧觨觩觪觫觬觭觮觯觰觱觲觳觞觵觯觷触觹觺觻觼觽觾觿"
u"言讠订讣訄訅訆訇计訉讯訋讧訍讨訏讦訑讱训訔讪讫托记訙訚讹訜讶訞讼訠訡䜣诀訤讷訦訧訨讻访訫訬设訮訯訰许訲訳诉訵诃訷訸訹诊注证訽訾訿"
u"詀诂詂詃詄詅诋詇詈詉詊詋詌詍讵詏诈詑诒詓诏评诐诇诎詙詚诅詜詝词詟咏诩询诣詤詥试詧詨诗詪诧诟诡诠詯诘话该详詴诜詶詷詸詹詺詻诙詽詾诖"
u"誀誁誂誃诔诛诓夸誈誉誊誋志认誎誏誐诳诶誓誔诞誖誗诱誙诮誛誜誝语誟诚诫誢诬误诰诵誧诲誩说誫说読誮誯谁誱课誳誴誵谇誷誸诽誺誻谊誽訚调"
u"諀諁谄諃谆諅諆谈諈诿諊请諌诤諎诹諐诼谅諓諔諕论谂諘諙諚谀谍谞谝諟諠谥诨諣谔諥谛谐諨諩諪谏諬谕谘諯諰讳諲谙諴諵谌讽诸諹谚諻谖諽诺諿"
u"谋谒谓謃誊诌謆謇謈謉谎謋謌謍谜謏谧謑謒謓谑謕谡谤謘谦谥讲謜谢謞謟谣谣謢謣謤謥謦謧谟謩謪谪谬谫謮謯謰謱謲讴謴謵謶謷謸谨謺謻謼謽谩謿"
u"譀哗譂譃譄䜧譆譇譈证譊譋譌譍谲讥譐譑譒譓譔譕谮譗识谯谭譛谱譝譞噪譠譡譢譣譤譥警譧譨譩譪谵譬譭譮译议譱譲譳谴譵譶护诪譹譺譻譼誉谫譿"
u"读讁讂讃讄讅讆讇讈讉变讋䜩讍雠讏讐讑谗让讔谰谶讗讘讙赞讛谠讝谳讟讠计订讣认讥讦讧讨让讪讫讬训议讯记讱讲讳讴讵讶讷许讹论讻讼讽设访"
u"诀证诂诃评诅识诇诈诉诊诋诌词诎诏诐译诒诓诔试诖诗诘诙诚诛诜话诞诟诠诡询诣诤该详诧诨诩诪诫诬语诮误诰诱诲诳说诵诶请诸诹诺读诼诽课诿"
u"谀谁谂调谄谅谆谇谈谉谊谋谌谍谎谏谐谑谒谓谔谕谖谗谘谙谚谛谜谝谞谟谠谡谢谣谤谥谦谧谨谩谪谫谬谭谮谯谰谱谲谳谴谵谶谷谸谹谺谻谼谽谾谿"
u"豀豁豂豃豄豅豆豇岂豉豊豋豌豍竖豏丰豑豒豓艳豕豖豗豘豙豚豛豜豝豞豟豠象豢豣豤豥豦豧豨豩豪豫猪豭豮豯豰豱豲豳豴豵豮豷豸豹豺豻豼豽豾豿"
u"貀貁貂貃貄貅貆貇貈貉貊貋貌狸貎貏貐貑貒猫貔貕貖貗貘䝙貚貛貜贝贞贠负财贡貣貤貥貦贫货贩贪贯责貭貮贮贳貱赀贰贵貵贬买贷貹贶费贴贻貾贸"
u"贺贲赂赁贿赅賆资贾賉贼賋賌賍賎賏賐赈赊宾賔赇賖賗賘赒赉賛赐賝赏賟赔赓贤卖贱賥赋赕賨賩质赍账赌賮賯䞐賱賲賳赖赗賶賷剩賹赚赙购赛赜賿"
u"贀贁贂贃贽赘贆赟赠贉赞赝贌赡贎赢赆贑贒赃赑贕赎赝贘贙贚赣赃贝贞负贠贡财责贤败账货质贩贪贫贬购贮贯贰贱贲贳贴贵贶贷贸费贺贻贼贽贾贿"
u"赀赁赂赃资赅赆赇赈赉赊赋赌赍赎赏赐赑赒赓赔赕赖赗赘赙赚赛赜赝赞赟赠赡赢赣赤赥赦赧赨赩赪赫赪赭赮赯走赱赲赳赴赵赶起赸赹赺赻赼赽赾赿"
u"趀趁趂趃趄超趆趇趈趉越趋趌趍趎趏趐趑趒趓趔赶趖趗趘赵趚趛趜趝趞趟趠趡趢趣趤趥趦趧趋趩趪趫趬趭趮趯趰趱趱足趴趵趶趷趸趹趺趻趼趽趾趿"
u"跀跁跂跃跄跅跆跇跈跉跊跋跌跍跎跏跐跑跒跓跔跕跖跗跘跙跚跛跜距跞跟跠迹跢跣交跥跦跧跨跩跪跫跬跭跮路跰跱跲跳跴践跶跷跸跹跺跻局跽跾跿"
u"踀踁踂踃踄踅踆踇踈踉踊踋踌踍踎踏践踑踒踓踔踕踖踗踘踙踚踛踜踝踞踟踠蜷踢踣踤踥踦踧踨踩踪踫踬踭踮踯逾踱踲踳踊踵踶踷踸踹踺踻踼踽踾踿"
u"蹀蹁蹂蹃蹄蹅蹆蹇蹈蹉蹊蹋跄蹍蹎蹏蹐蹑蹒蹓蹔跸蹖蹗蹘蹙蹚蹛蹜蹝蹞迹蹠蹡蹢蹒踪蹥蹦糟蹨蹩蹪蹫蹬蹭蹮蹯蹰蹱蹲蹳蹴蹵蹶蹷蹸蹹跷蹻蹼蹽蹾蹿"
u"躀躁跶躃躄躅躆躇躈趸踌跻躌跃躎躏躐踯跞踬躔蹰躖躗躘躙跹躛躜躝躞躟躠蹑躢躣躤蹿躜躧躨躩躏身躬躭躮躯躰躱躲躳躴躵躶躷躸躹躺躻躼躽躾躿"
u"躯軁軂軃軄軅軆軇軈軉车轧轨军軎軏軐轪轩軓轫軕軖軗軘軙軚轭軜軝軞软軠軡転軣轷軥軦軧軨軩軪轸軬軭軮軯軰軱轱軳軴軵軶軷轴轵轺轲轶軽轼軿"
u"輀輁輂较輄辂輆辁辀载轾輋輌輍輎輏輐輑辄挽辅轻輖輗輘輙輚辆辎辉辋辍輠輡輢輣輤辊辇輧輨辈轮輫辌輭輮辑輰輱輲辏輴輵輶輷输輹輺辐輼輽辗舆"
u"辒轁毂轃辖辕辘轇轈转轊轋轌辙轿轏轐轑轒轓辚轕轖轗轘轙轚轛轜舆轞轰轠辔轹轣轳轥车轧轨轩轪轫转轭轮软轰轱轲轳轴轵轶轷轸轹轺轻轼载轾轿"
u"辀辁辂较辄辅辆辇辈辉辊辋辌辍辎辏辐辑辒输辔辕辖辗辘辙辚辛辜辝辞辟辠辡辢辣辤辥办辧辨辩辪辫辬辞辫辩辰辱农辳辴辵辶辷辸边辺辻込辽达辿"
u"迀迁迂迃迄迅迆过迈迉迊迋迌迍迎迏运近迒迓返迕迖迗还这迚进远违连迟迠迡迢迣迤迥迦迧迨迩迪迫迬迭迮迯述迱迲迳回迵迶迷迸迹迺迻迼追迾迿"
u"退送适逃逄逅逆逇逈选逊逋逌逍逎透逐逑递逓途迳逖逗逘这通逛逜逝逞速造逡逢连逤逥逦逧逨逩逪逫逬逭逮逯逰周进逳逴逵逶逷逸逹逺逻逼逽逾逿"
u"遀遁遂遃遄遅遆遇遈遉游运遌遍过遏遐遑遒道达违遖遗遘遥遚遛逊遝递遟远遡遢遣遤遥遦遧遨适遪遫遬遭遮遁遰遱迟遳遴遵遶迁选遹遗遻辽遽遾避"
u"邀迈邂邃还邅邆迩邈邉边邋邌邍邎逻逦邑邒邓邔邕邖邗邘邙邚邛邜邝邞邟邠邡邢那邤邥邦邧邨邩邪邫邬邭邮邯邰邱邲邳邴邵邶邷邸邹邺邻邼邽邾邿"
u"郀郁郂郃郄郅郆郇郈郉郊郋郌郍郎郏郐郑郒郓郔郕郖郗郘郙郚郛郜郝郞郏郠郡郢郣郤郥郦郧部郩郪郫郬郭郮郯郰郱郲郳郴邮郶郷郸郹郺郻郼都郾郿"
u"鄀鄁鄂鄃鄄鄅郓鄇鄈乡鄊鄋鄌鄍鄎鄏鄐鄑邹鄓邬鄕郧鄗鄘鄙鄚鄛鄜鄝鄞鄟鄠鄡鄢鄣鄤鄥鄦邓鄨鄩鄪鄫鄬郑鄮鄯邻鄱郸鄳邺鄵郐鄷鄸鄹邝鄻鄼鄽鄾鄿"
u"酀酁酂酃酄酅酆酂郦酉酊酋酌配酎酏酐酑酒酓酔酕酖酗酘酙酚酛酜酝酞酟酠酡酢酣酤酥酦酧酨酩酪酫酬酭酮酯酰酱酲酳酴酵酶酷酸酹酺酻酼酽酾酿"
u"醀醁醂腌醄醅醆醇醈醉醊醋醌醍醎醏醐醑醒醓醔醕酝醗醘醙醚醛丑醝酝醟醠醡醢醣醤醥醦醧醨醩醪医酱醭醮醯醰酦醲醳醴醵醶醷醸醹醺醻宴醽醾醿"
u"酿衅釂酾釄酽釆采釈釉释释里重野量厘金钅钆钇钌釖钊钉钋釚釛釜针釞釟釠釡釢钓钐釥扣钏釨钒釪釫釬釭釮釯釰釱釲釳釴钗釶钍釸钕钎釻釼釽釾釿"
u"钯钫鈂钘钭鈅鈆鈇钚钠鈊鈋鈌钝钩鈏钤钣钑鈓钞钮鈖鈗鈘鈙鈚鈛鈜鈝钧鈟鈠鈡鈢钙鈤钬钛钪鈨鈩鈪鈫鈬鈭铌鈯铈鈱鈲钶铃鈵鈶钴钹铍钰鈻鈼钸铀钿"
u"钾鉁鉂鉃鉄钜鉆鉇铊铉鉊铇鉌铋鉎鉏鉐铂鉒鉓鉔钷鉖钳鉘鉙铆铅鉜鉝钺鉟鉠鉡钵鉣钩鉥钲鉧鉨鉩鉪鉫钼钽鉮鉯鉰鉱鉲鉳鉴鉵铏鉷铰鉹铒铬鉼鉽鉾铪"
u"银銁銂铳銄铜銆銇銈銉銊銋銌铚銎銏銐铣銒铨銔銕铢銗铭銙铫铦衔銝銞銟铑銡銢铷銤铱铟銧铵铥铕铯铐銭銮銯銰铞焊锐銴銵銶销銸锈銺锑锉銽銾銿"
u"鋀铝鋂锒鋄锌鋆钡鋈鋉鋊鋋铤鋍鋎铗鋐鋑锋鋓鋔鋕鋖鋗鋘铻鋚鋛鋜锊鋞锓鋠鋡鋢铘锄锃锔鋧锇铓铺鋫鋬锐铖锆锂铽鋲鋳鋴鋵锍鋷锯鋹鋺鋻钢鋽鋾鋿"
u"錀锞錂錃录錅锖锫锩錉錊錋錌錍錎铔锥錑锕錓錔锟錖錗锤锱铮锛錜錝錞锬锭锜钱錣錤錥锦錧锚锠錪锡錬錭锢错錰錱录锰錴錵表錷铼錹錺錻錼錽錾錿"
u"锝锨鍂锪鍄鍅钔锴锳鍉炼锅鍌镀鍎鍏鍐鍑鍒鍓锷鍕鍖鍗铡鍙钖锻鍜鍝鍞鍟锽鍡鍢鍣锸锲鍦鍧鍨锘鍪鍫锹鍭鍮鍯锾鍱鍲鍳鍴键锶鍷鍸鍹锗鍻鍼鍽钟鍿"
u"鎀鎁镁鎃锿鎅鎆镅鎈鎉镑鎋鎌鎍鎎鎏鎐鎑鎒鎓镕鎕锁枪镉鎙锤镈鎜鎝鎞鎟鎠镃钨蓥鎤鎥镏铠鎨铩锼鎫镐鎭镇鎯镒鎱镋镍鎴镓鎶鎷镌鎹鎺鎻鎼鎽鎾镎"
u"鏀鏁鏂镞鏄鏅鏆镟链鏉鏊鏋镆镙鏎鏏镠镝鏒鏓鏔鏕鏖铿锵鏙鏚鏛镗镘镛铲鏠镜镖鏣镂鏥鏦鏧錾鏩鏪鏫鏬鏭鏮鏯镚鏱鏲鏳鏴铧鏶镤鏸镪鏺鏻鏼锈鏾鏿"
u"鐀鐁鐂铙鐄鐅鐆鐇鐈鐉鐊铴鐌鐍鐎鐏镣鐑铹镦镡鐕鐖鐗钟镫鐚鐛鐜镢鐞鐟镨鐡鐢鐣鐤鐥锎锏镄鐩鐪镌鐬鐭镰鐯鐰鐱镯镭鐴铁镮鐷铎鐹铛鐻鐼鐽鐾镱"
u"鑀鑁鑂鑃铸鑅鑆鑇鑈鑉镬鑋镔鑍鑎鑏鑐鉴鉴鑓镲锧鑖鑗鑘鑙鑚鑛鑜鑝镴鑟铄鑡鑢镳鑤镥鑦鑧鑨鑩鑪鑫鑬镧鑮鑯钥镵镶鑳鑴鑵鑶镊鑸镩鑺鑻锣钻銮凿"
u"钀䦆钂钃钄钅钆钇针钉钊钋钌钍钎钏钐钑钒钓钔钕钖钗钘钙钚钛钜钝钞钟钠钡钢钣钤钥钦钧钨钩钪钫钬钭钮钯钰钱钲钳钴钵钶钷钸钹钺钻钼钽钾钿"
u"铀铁铂铃铄铅铆铇铈铉铊铋铌铍铎铏铐铑铒铓铔铕铖铗铘铙铚铛铜铝铞铟铠铡铢铣铤铥铦铧铨铩铪铫铬铭铮铯铰铱铲铳铴铵银铷铸铹铺铻铼铽链铿"
u"销锁锂锃锄锅锆锇锈锉锊锋锌锍锎锏锐锑锒锓锔锕锖锗锘错锚锛锜锝锞锟锠锡锢锣锤锥锦锧锨锩锪锫锬锭键锯锰锱锲锳锴锵锶锷锸锹锺锻锼锽锾锿"
u"镀镁镂镃镄镅镆镇镈镉镊镋镌镍镎镏镐镑镒镓镔镕镖镗镘镙镚镛镜镝镞镟镠镡镢镣镤镥镦镧镨镩镪镫镬镭镮镯镰镱镲镳镴镵镶长镸镹镺镻镼镽镾长"
u"门閁闩闪閄閅闫閇闬闭閊开闶閍闳闰閐闲闲间闵閕閖閗闸閙閚閛閜閝閞閟閠阂关阁閤阀閦哄闺闽閪阃阆闾閮閯閰阅阅閳閴閵阊閷閸阉閺阎阏阍阈阌"
u"闀闁闂阒闄闅板暗闱闉阔阕阑阇闎闏阗闑阘闿阖阙闯闗斗闙闚闛关闝阚闟阓阐辟闣阛闼闦闧门闩闪闫闬闭问闯闰闱闲闳间闵闶闷闸闹闺闻闼闽闾闿"
u"阀阁阂阃阄阅阆阇阈阉阊阋阌阍阎阏阐阑阒阓阔阕阖阗阘阙阚阛阜阝阞队阠阡阢阣阤阥阦阧厄阩坂阫阬阭阮阯阰阱防阳阴阵阶阷阸阹阺阻阼阽阾阿"
u"陀陁陂陃附际陆陇陈陉陊陋陌降陎陏限陑陒陓陔陕陖陗陉陙陚陛陜陕升陟陠陡院阵除陥陦陧陨险陪陫陬陭陮陯阴陱陲陈陴陵陶陷陆陹険陻陼阳陾陿"
u"隀隁隂隃堤隅隆隇隈陧队隋隌隍阶随隐隑隒隓隔陨隖隗隘隙隚际障隝隞隟隠隡隢隣隤隥隦隧随隩险隫隬隭隮隯隰隐隲隳陇隵隶隷隶隹隺只隼隽难隿"
u"雀雁雂雃雄雅集雇雈雉雊隽雌雍雎雏雐雑雒雓雔雕虽雗雘双雚雏杂雝鸡雟雠雡离难雤雥雦雧雨雩雪雫雬雭雮雯雰雱云雳雴雵零雷雸雹雺电雼雽雾雿"
u"需霁霂霃霄霅霆震霈霉霊霋霌霍霎霏霐沾霒霓霔霕霖霗霘霙霚霛霜霝霞霟霠霡霡霣霤霥霦雾霨霩霪霫霬霭霮霯霰霱露霳霴霵霶霷霸霹霺霻霼霁霾霿"
u"靀靁雳靃霭靅靆靇灵靉靊靋靌靍靎靏靐靑青靓靔靕靖靗靘静靓靛静靝非靟靠靡面靣靤靥腼靧靥革靪靫靬靭靮靯靰靱靲靳靴靵靶纼靸靹靺靻靼靽靾靿"
u"鼗鞁鞂鞃鞄鞅鞆鞇鞈鞉鞊鞋鞌鞍鞎巩鞐鞑鞒鞓鞔鞕鞖鞗鞘鞙鞚鞛鞜绱鞞鞟鞠鞡鞢鞣鞤鞥鞦鞧鞨鞩鞪鞫鞬鞭鞮鞯鞰鞱鞲鞳鞴鞵鞶鞷鞸鞹鞺鞻鞼鞒鞾鞿"
u"韀缰韂鞑韄韅韆韇韈鞯韊韦韧韨韎韏韐韑韒韩韔韕韖韗韘韪韚韛韬韝韫韟韠韡韢韣韤韥韦韧韨韩韪韫韬韭韭韯韰韱韲音韴韵韶韷韸韹韺韵韼韽韾响"
u"頀页顶顷頄项顺顸须頉顼頋颂頍颀颃预顽颁顿頔頕頖颇领頙頚頛颌頝頞頟頠颉頢頣颐頥颏頧頨頩頪頫頬头颒頯颊頱颋頳颕頵頶颔颈颓頺频頼颓頾頿"
u"顀顁顂顃顄顅颗顇顈顉顊顋题额颚颜顐顑颙颛颜顕顖顗愿颡顚颠顜顝类顟顠顡颟顣顤颢顦顾顨顩顪颤颥顭顮显颦颅顲颞颧页顶顷顸项顺须顼顽顾顿"
u"颀颁颂颃预颅领颇颈颉颊颋颌颍颎颏颐频颒颓颔颕颖颗题颙颚颛颜额颞颟颠颡颢颣颤颥颦颧风颩颪颫颬飐飑飒颰台颲刮颴颵飓颷飔颹飏飖飕颽颾颿"
u"飗飁飂飃飘飅飙飇飚飉飊飋飌飍风飏飐飑飒飓飔飕飖飗飘飙飚飞飜飝飞食饣飡饥饤飤饦飦飧飨饨饪饫飬饬飮饭飰飱饮飳饴飵飶飷飸飹飺飻饲饱饰饳"
u"餀餁餂饺饸饼餆餇餈饷养餋饵餍饹饻餐饽馁饿餔馂饾餗余餙肴馄馃餝饯餟餠馅餢餣餤餥餦餧馆餩餪餫糊餭餮餯餰糇餲饧餴喂馉馇餸餹馎餻饩馈馏馊"
u"饀馌饂馍饄馒饆饇馐馑馓馈馔饍饎饏饐饥饶饓饔饕饖飨饘饙饚饛餍饝馋饟饠饡馕饣饤饥饦饧饨饩饪饫饬饭饮饯饰饱饲饳饴饵饶饷饸饹饺饻饼饽饾饿"
u"馀馁馂馃馄馅馆馇馈馉馊馋馌馍馎馏馐馑馒馓馔馕首馗馘香馚馛馜馝馞馟馠馡馢馣馤馥馦馧馨馩馪馫马驭冯馯馰驮馲驰驯馵馶馷馸驲馺馻馼馽馾馿"
u"駀驳駂駃駄駅駆駇駈駉駊駋駌駍駎駏驻驽驹駓驵驾駖駗骀驸駚驶駜驼駞驷駠骂骈駣駤駥駦駧駨駩駪駫駬骇駮駯骃骆駲駳駴駵駶駷骎駹駺駻駼駽駾骏"
u"騀骋骍騃騄骓騆騇騈騉騊騋骔骒骑骐騐騑騒験騔騕骛騗騘骗騚騛騜騝騞騟騠騡騢騣骙騥騦䯄騨騩騪骞騬骘骝騯腾騱騲騳騴騵驺骚骟騹騺騻騼騽骡騿"
u"蓦骜骖骠骢驱驆驇驈驉骅驋骕骁驎骣驐驑驒驓驔骄驖验驘驙惊驿驜驝驞骤驠驡驴驣骧骥骦驧驨驩骊骉马驭驮驯驰驱驲驳驴驵驶驷驸驹驺驻驼驽驾驿"
u"骀骁骂骃骄骅骆骇骈骉骊骋验骍骎骏骐骑骒骓骔骕骖骗骘骙骚骛骜骝骞骟骠骡骢骣骤骥骦骧骨骩骪骫骬骭骮肮骰骱骲骳骴骵骶骷骸骹骺骻骼骽骾骿"
u"髀髁髂髃髄髅髆髇髈髉髊髋髌髍髎髅髐髑脏髓体髌髋髗高髙髚髛髜髝髞髟髠髡髢髣髤髥髦髧髨髩髪髫髬髭发髯髰髱髲髳髴髵髶髷髸髹髺髻髼髽髾髿"
u"剃鬁鬂鬃鬄鬅松鬇鬈鬉鬊鬋鬌胡鬎鬏鬐鬑鬒鬓鬔鬕鬖鬗鬘鬙须鬛鬜鬝鬞鬟鬠鬡鬓鬣鬤斗鬦闹哄阋鬪鬫鬬斗阄鬯鬰郁鬲鬳鬴鬵鬶鬷鬸鬹鬺鬻鬼鬽鬾鬿"
u"魀魁魂魃魄魅魆魇魈魉魊魋魌魍魉魏魐魑魒魓魔魕魖魗魇魙鱼鱽魜魝魞魟魠魡鱾魣魤魥魦魧鲀魩魪魫魬魭魮鲁魰魱魲魳鲂魵魶鱿魸魹鲄魻魼魽魾魿"
u"鮀鲅鮂鲆鮄鮅鮆鮇鮈鮉鲌鲉鮌鲏鲇鮏鲐鲍鲋鲊鮔鮕鮖鮗鮘鮙鲒鮛鲘鲞鲕鮟鮠鮡鮢鮣鮤鮥鲖鮧鮨鮩鲔鲛鮬鲑鲜鮯鮰鮱鮲鲓鮴鮵鲪鮷鮸鮹鲝鮻鮼鮽鮾鮿"
u"鲧鲠鯂鯃鯄鯅鯆鲩鯈鲤鲨鯋鯌鯍鯎鯏鯐鯑鲬鯓鲻鲯鲭鯗鯘鯙鯚鲷鯜鲴鯞鯟鯠鲱鲵鯣鲲鯥鯦鲳鲸鯩鲮鲰鯬鯭鯮鯯鲶鯱鯲鯳鲺鯵鯶鳀鯸鯹鯺鯻鯼鲫鯾鳊"
u"鰀鳈鲗鳂鰄鰅鰆鰇鲽鳇鰊鰋鰌鳅鰎鲾鳄鰑鳆鳃鰔鰕鰖鰗鰘鰙鰚鰛鳒鰝鰞鳑鳋鰡鰢鲥鰤鳏鰦鰧鳎鳐鰪鰫鰬鳍鳁鰯鰰鲢鳌鳓鰴鳘鰶鲦鰸鲣鲹鳗鳛鰽鳔鰿"
u"鱀鱁鳉鱃鱄鳙鱆鱇鳕鳖鱊鱋鱌鱍鱎鱏鱐鱑鳟鱓鳝鱕鳜鳞鲟鱙鱚鱛鱜鲼鱞鲎鲙鱡鱢鳣鳡鱥鱦鳢鲿鱩鱪鱫鱬鲚鱮鳠鱰鱱鱲鱳鱴鱵鱶鳄鲈鱹鲡鱻鱼鱽鱾鱿"
u"鲀鲁鲂鲃鲄鲅鲆鲇鲈鲉鲊鲋鲌鲍鲎鲏鲐鲑鲒鲓鲔鲕鲖鲗鲘鲙鲚鲛鲜鲝鲞鲟鲠鲡鲢鲣鲤鲥鲦鲧鲨鲩鲪鲫鲬鲭鲮鲯鲰鲱鲲鲳鲴鲵鲶鲷鲸鲹鲺鲻鲼鲽鲾鲿"
u"鳀鳁鳂鳃鳄鳅鳆鳇鳈鳉鳊鳋鳌鳍鳎鳏鳐鳑鳒鳓鳔鳕鳖鳗鳘鳙鳚鳛鳜鳝鳞鳟鳠鳡鳢鳣鳤鸟鳦凫鳨鸠鳪鳫凫鳭鳮鳯鳰鳱鸤凤鸣鳵鸢鳷鳸鳹鳺鳻鳼鳽䴓鳿"
u"鴀鴁鴂鴃鴄鴅鸩鸨鴈鸦鴊鴋鴌鴍鴎鴏鴐鴑鸰鴓鴔鸵鴖鴗鴘鴙鴚鸳鴜鸲鸮鸱鴠鴡鴢鸪鴤鴥鸯鴧鸭鴩鴪鴫鴬鴭鴮鸸鸹鴱鴲鴳鸻鴵鴶䴕鴸鴹鴺鸿鴼鴽鴾鸽"
u"鵀䴔鸺鸼鵄鵅鵆鵇鵈鵉鵊鵋鵌鵍鵎鵏鹀鹃鹆鹁鵔鵕鵖鵗鵘鵙鵚鵛鹈鹅鵞鵟鹄鹉鵢鵣鵤鵥鵦鵧鵨鵩鹌鵫鹏鵭鹐鹎鵰鵱鹊鵳鵴鵵鵶鹓鵸鵹鵺鵻鵼鵽鹍鵿"
u"鶀鶁鶂鶃䴖鶅鶆鸫鶈鹑鹒鶋鶌鶍鶎鶏鶐鶑鶒鹋鶔鶕鹙鶗鹕鶙鹗鶛鶜鶝鶞鶟鶠鹖鶢鶣鶤鹛鶦鶧鶨鹜䴗鶫鸧鶭鶮莺鶰鶱鹟鶳鹤鶵鶶鶷鶸鹠鹡鹘鹣鶽鶾鶿"
u"鹚鹢鹞鷃鸡鷅鷆鷇䴘鷉鹝鷋鷌鷍鷎鷏鷐鷑鷒鹧鷔鷕鹥鸥鷘鸷鹨鷛鷜鷝鷞鷟鷠鷡鷢鷣鷤鸶鹪鷧鷨鷩鷪鹔鷬鷭鷮鹩鷰鷱鹫鹇鷴鷵鷶鷷鹬鹰鹭鷻鷼鸴鷾䴙"
u"鸀鸁㶉鸃鸄鸅鸆鹯鸈鸉鸊鸋鹱鸍鸎鹲鸐鸑鸒鸓鸔鸬鸖鸗鹴鸙鹦鹳鸜鹂鸾鸟鸠鸡鸢鸣鸤鸥鸦鸧鸨鸩鸪鸫鸬鸭鸮鸯鸰鸱鸲鸳鸴鸵鸶鸷鸸鸹鸺鸻鸼鸽鸾鸿"
u"鹀鹁鹂鹃鹄鹅鹆鹇鹈鹉鹊鹋鹌鹍鹎鹏鹐鹑鹒鹓鹔鹕鹖鹗鹘鹙鹚鹛鹜鹝鹞鹟鹠鹡鹢鹣鹤鹥鹦鹧鹨鹩鹪鹫鹬鹭鹮鹯鹰鹱鹲鹳鹴卤鹶鹷鹸咸鹾鹻碱盐鹾鹿"
u"麀麁麂麃麄麅麆麇麈麉麊麋麌麍麎麏麐麑麒麓麔麕麖丽麘麙麚麛麜麝麞麟麠麡麢麣粗麦麦麧麨麸麪麫麬麭麮曲麰麱麲麳麴面麶麷麸麹麺麻么么麾麿"
u"黀黁黂黄黄黅黆黇黈黉黊黋黉黍黎黏黐黑黒黓黔黕黖黗默黙黚黛黜黝点黟黠黡黢黣黤黥黦黧党黩黪黫黬黭黮黯黰黱黪黳霉黵黡黩黸黹黺黻黼黾黾鼋"
u"鼀鼁鼂鼃鼄鼅鼆鳌鳖鼍鼊鼋鼌鼍鼎鼏鼐鼑鼒鼓鼔冬鼖鼗鼘鼙鼚鼛鼜鼝鼞鼟鼠鼡鼢鼣鼤鼥鼦鼧鼨鼩鼪鼫鼬鼭鼮鼯鼰鼱鼲鼳鼹鼵鼶鼷鼸鼹鼺鼻鼼鼽鼾鼿"
u"齀齁齂齃齄齅齆齇齈齉齐斋齌齍赍齑齐齑齿齓龀龁齖龂齘龅齚齛龇齝齞龃龆龄齢出齤齥龈啮齨咬龊齫龉齭齮齯齰齱龋齳齴齵腭龌齸齹齺齻齼齽齾齿"
u"龀龁龂龃龄龅龆龇龈龉龊龋龌龙厐龏庞龑龒龓龚龛龖龗龘龙龚龛龟龝龞龟龠龡龢龣龤龥";

